<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\BusinessChangeRequestModel;
use App\Models\StoreModel;
use App\Models\BusinessUserModel;
use App\Services\BusinessEmailService;
use CodeIgniter\HTTP\ResponseInterface;

class ChangeRequests extends BaseController
{
    protected $changeRequestModel;
    protected $storeModel;
    protected $businessUserModel;
    protected $emailService;

    public function __construct()
    {
        $this->changeRequestModel = new BusinessChangeRequestModel();
        $this->storeModel = new StoreModel();
        $this->businessUserModel = new BusinessUserModel();
        $this->emailService = new BusinessEmailService();
    }

    /**
     * Display all change requests with filtering
     */
    public function index()
    {
        // Check admin authentication
        if (!$this->isAdminLoggedIn()) {
            return redirect()->to('/admin/login');
        }

        // Get filter parameters
        $status = $this->request->getGet('status') ?? 'all';
        $storeId = $this->request->getGet('store');
        $requestType = $this->request->getGet('type');

        // Build query based on filters
        $requests = $this->getFilteredRequests($status, $storeId, $requestType);

        // Get statistics for dashboard
        $stats = $this->changeRequestModel->getAdminStats();

        // Get unique stores for filter dropdown
        $stores = $this->storeModel->select('store_id, name')
                                 ->orderBy('name', 'ASC')
                                 ->findAll();

        $data = [
            'title' => 'Business Change Requests',
            'requests' => $requests,
            'stats' => $stats,
            'stores' => $stores,
            'current_status' => $status,
            'current_store' => $storeId,
            'current_type' => $requestType,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('admin/change_requests/index', $data);
    }

    /**
     * View detailed change request for review
     */
    public function review($requestId)
    {
        if (!$this->isAdminLoggedIn()) {
            return redirect()->to('/admin/login');
        }

        // Get change request with full details
        $request = $this->changeRequestModel->getRequestForReview($requestId);

        if (!$request) {
            return redirect()->to('/admin/change-requests')->with('error', 'Change request not found.');
        }

        // Get current store data for comparison
        $currentStore = $this->storeModel->find($request['store_id']);

        if (!$currentStore) {
            return redirect()->to('/admin/change-requests')->with('error', 'Associated business not found.');
        }

        // Decode field changes
        $fieldChanges = json_decode($request['field_changes'], true);

        // Process changes for display
        $processedChanges = $this->processChangesForDisplay($fieldChanges, $request['request_type']);

        $data = [
            'title' => 'Review Change Request',
            'request' => $request,
            'current_store' => $currentStore,
            'field_changes' => $fieldChanges,
            'processed_changes' => $processedChanges,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('admin/change_requests/review', $data);
    }

    /**
     * Approve change request and apply changes
     */
    public function approve($requestId)
    {
        if (!$this->isAdminLoggedIn()) {
            return redirect()->to('/admin/login');
        }

        $adminId = $this->getAdminId();
        $adminNotes = $this->request->getPost('admin_notes');

        // Get the change request
        $request = $this->changeRequestModel->find($requestId);

        if (!$request || $request['status'] !== 'pending') {
            return redirect()->back()->with('error', 'Invalid change request or already processed.');
        }

        // Start transaction
        $db = \Config\Database::connect();
        $db->transStart();

        try {
            // Apply changes to store table
            $fieldChanges = json_decode($request['field_changes'], true);
            $updateData = [];

            foreach ($fieldChanges as $field => $change) {
                if ($request['request_type'] === 'hours_update') {
                    // Handle hours updates differently (more complex)
                    $this->applyHoursChanges($request['store_id'], $fieldChanges);
                } else {
                    // Regular field updates
                    $updateData[$field] = $change['new'];
                }
            }

            // Update store data if not hours update
            if (!empty($updateData) && $request['request_type'] !== 'hours_update') {
                $this->storeModel->update($request['store_id'], $updateData);
            }

            // Approve the request
            $this->changeRequestModel->approveRequest($requestId, $adminId, $adminNotes);

            $db->transComplete();

            if ($db->transStatus() === false) {
                return redirect()->back()->with('error', 'Failed to approve change request. Database transaction failed.');
            }

            // Send notification to business user
            $this->emailService->notifyBusinessApproval($requestId, $adminNotes);

            return redirect()->to('/admin/change-requests')->with('success', 'Change request approved and changes applied successfully.');

        } catch (\Exception $e) {
            $db->transRollback();
            log_message('error', 'Error approving change request: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to approve change request. Please try again.');
        }
    }

    /**
     * Reject change request
     */
    public function reject($requestId)
    {
        if (!$this->isAdminLoggedIn()) {
            return redirect()->to('/admin/login');
        }

        $adminId = $this->getAdminId();
        $adminNotes = $this->request->getPost('admin_notes');

        if (empty($adminNotes)) {
            return redirect()->back()->with('error', 'Admin notes are required when rejecting a request.');
        }

        // Get the change request
        $request = $this->changeRequestModel->find($requestId);

        if (!$request || $request['status'] !== 'pending') {
            return redirect()->back()->with('error', 'Invalid change request or already processed.');
        }

        // Reject the request
        if ($this->changeRequestModel->rejectRequest($requestId, $adminId, $adminNotes)) {
            // Send notification to business user
            $this->emailService->notifyBusinessRejection($requestId, $adminNotes);

            return redirect()->to('/admin/change-requests')->with('success', 'Change request rejected successfully.');
        } else {
            return redirect()->back()->with('error', 'Failed to reject change request. Please try again.');
        }
    }

    /**
     * Bulk approve multiple requests
     */
    public function bulkApprove()
    {
        if (!$this->isAdminLoggedIn()) {
            return redirect()->to('/admin/login');
        }

        $requestIds = $this->request->getPost('request_ids');
        $adminNotes = $this->request->getPost('bulk_admin_notes');

        if (empty($requestIds) || !is_array($requestIds)) {
            return redirect()->back()->with('error', 'No requests selected for bulk approval.');
        }

        $adminId = $this->getAdminId();
        $successCount = 0;
        $errorCount = 0;

        foreach ($requestIds as $requestId) {
            try {
                $request = $this->changeRequestModel->find($requestId);
                
                if ($request && $request['status'] === 'pending') {
                    // Apply changes and approve
                    $fieldChanges = json_decode($request['field_changes'], true);
                    
                    if ($request['request_type'] !== 'hours_update') {
                        $updateData = [];
                        foreach ($fieldChanges as $field => $change) {
                            $updateData[$field] = $change['new'];
                        }
                        $this->storeModel->update($request['store_id'], $updateData);
                    } else {
                        $this->applyHoursChanges($request['store_id'], $fieldChanges);
                    }

                    $this->changeRequestModel->approveRequest($requestId, $adminId, $adminNotes);
                    $this->emailService->notifyBusinessApproval($requestId, $adminNotes);
                    $successCount++;
                }
            } catch (\Exception $e) {
                $errorCount++;
                log_message('error', 'Bulk approve error for request ' . $requestId . ': ' . $e->getMessage());
            }
        }

        $message = "Bulk approval complete. {$successCount} requests approved";
        if ($errorCount > 0) {
            $message .= ", {$errorCount} failed";
        }

        return redirect()->back()->with('success', $message . '.');
    }

    /**
     * Get pending requests count for admin dashboard
     */
    public function getPendingCount()
    {
        if (!$this->isAdminLoggedIn()) {
            return $this->response->setJSON(['error' => 'Unauthorized']);
        }

        $count = $this->changeRequestModel->where('status', 'pending')->countAllResults();
        return $this->response->setJSON(['pending_count' => $count]);
    }

    /**
     * Apply hours changes to store schedules
     */
    private function applyHoursChanges($storeId, $hoursChanges)
    {
        // This would integrate with your existing hours management system
        // For now, we'll create a placeholder that logs the hours changes
        // You would replace this with your actual hours update logic
        
        foreach ($hoursChanges as $day => $change) {
            if (isset($change['new'])) {
                // Update store_schedules table with new hours
                log_message('info', "Updating {$day} hours for store {$storeId}: " . json_encode($change['new']));
                // TODO: Implement actual hours update logic based on your store_schedules structure
            }
        }
    }

    /**
     * Get filtered requests based on criteria
     */
    private function getFilteredRequests($status, $storeId, $requestType)
    {
        $builder = $this->changeRequestModel->select('business_change_requests.*, business_users.first_name, business_users.last_name, business_users.username, store.name as store_name')
                                           ->join('business_users', 'business_users.id = business_change_requests.business_user_id', 'left')
                                           ->join('store', 'store.store_id = business_change_requests.store_id', 'left');

        if ($status !== 'all') {
            $builder->where('business_change_requests.status', $status);
        }

        if ($storeId) {
            $builder->where('business_change_requests.store_id', $storeId);
        }

        if ($requestType) {
            $builder->where('business_change_requests.request_type', $requestType);
        }

        return $builder->orderBy('business_change_requests.requested_at', 'DESC')
                      ->paginate(20);
    }

    /**
     * Process changes for admin display
     */
    private function processChangesForDisplay($fieldChanges, $requestType)
    {
        $processed = [];

        foreach ($fieldChanges as $field => $change) {
            $processed[$field] = [
                'field_label' => $this->getFieldLabel($field),
                'old_value' => $change['old'] ?? '',
                'new_value' => $change['new'] ?? '',
                'has_change' => ($change['old'] ?? '') !== ($change['new'] ?? '')
            ];
        }

        return $processed;
    }

    /**
     * Get field labels for display
     */
    private function getFieldLabel($field)
    {
        $labels = [
            'name' => 'Business Name',
            'description' => 'Description',
            'email' => 'Email Address',
            'phone' => 'Phone Number',
            'website' => 'Website',
            'address' => 'Street Address',
            'city' => 'City',
            'state' => 'State',
            'postal_code' => 'Zip Code',
            'facebook' => 'Facebook',
            'instagram' => 'Instagram',
            'x_twitter' => 'X (Twitter)',
            'linkedin' => 'LinkedIn',
            'youtube' => 'YouTube',
            'tiktok' => 'TikTok',
            'snapchat' => 'Snapchat',
            'google_business' => 'Google Business',
            'yelp' => 'Yelp',
            'nextdoor' => 'Nextdoor',
            'additional_info' => 'Additional Info (Vimeo ID)'
        ];

        return $labels[$field] ?? ucfirst(str_replace('_', ' ', $field));
    }

    /**
     * Test email configuration
     */
    public function testEmail()
    {
        if (!$this->isAdminLoggedIn()) {
            return redirect()->to('/admin/login');
        }

        $result = $this->emailService->sendTestEmail();

        if ($result['success']) {
            return redirect()->back()->with('success', 'Test email sent successfully! Check your inbox.');
        } else {
            return redirect()->back()->with('error', 'Email test failed: ' . $result['message']);
        }
    }

    /**
     * Send notification to business user
     */
    private function sendBusinessNotification($request, $status, $adminNotes)
    {
        // This method is now handled by the EmailService
        // Keeping for backward compatibility, but using the service
        if ($status === 'approved') {
            return $this->emailService->notifyBusinessApproval($request['id'], $adminNotes);
        } elseif ($status === 'rejected') {
            return $this->emailService->notifyBusinessRejection($request['id'], $adminNotes);
        }
        
        return false;
    }

    /**
     * Check if admin is logged in
     */
    private function isAdminLoggedIn()
    {
        return session()->get('admin_logged_in') === true;
    }

    /**
     * Get current admin ID
     */
    private function getAdminId()
    {
        return session()->get('admin_id');
    }
}