<?php

namespace App\Controllers\Admin;

use App\Controllers\BaseController;
use App\Models\BusinessJobModel;
use App\Models\StoreModel;
use App\Models\BusinessUserModel;

class Jobs extends BaseController
{
    protected $jobModel;
    protected $storeModel;
    protected $businessUserModel;
    protected $session;

    public function __construct()
    {
        $this->jobModel = new BusinessJobModel();
        $this->storeModel = new StoreModel();
        $this->businessUserModel = new BusinessUserModel();
        $this->session = \Config\Services::session();
    }

    /**
     * Main admin jobs management page
     */
    public function index()
    {
        if (!$this->isAdminLoggedIn()) {
            return redirect()->to('/admin/auth/login');
        }

        // Get filter parameters
        $filter = $this->request->getGet('filter') ?? 'active';
        $search = $this->request->getGet('search') ?? '';
        $employmentType = $this->request->getGet('employment_type') ?? '';
        $business = $this->request->getGet('business') ?? '';
        $page = (int)($this->request->getGet('page') ?? 1);
        $perPage = 20;

        // Build query based on filters
        $builder = $this->jobModel->select('business_jobs.*, store.name as store_name, store.city, store.state, store.phone, store.email, business_users.first_name, business_users.last_name, business_users.email as user_email')
                                 ->join('store', 'store.store_id = business_jobs.store_id')
                                 ->join('business_users', 'business_users.id = business_jobs.created_by', 'left');

        // Apply filters
        switch ($filter) {
            case 'active':
                $builder->where('business_jobs.is_active', 1)
                       ->where('business_jobs.expires_at >', date('Y-m-d H:i:s'));
                break;
            case 'expiring':
                $expiringDate = date('Y-m-d H:i:s', strtotime('+7 days'));
                $builder->where('business_jobs.is_active', 1)
                       ->where('business_jobs.expires_at <=', $expiringDate)
                       ->where('business_jobs.expires_at >', date('Y-m-d H:i:s'));
                break;
            case 'expired':
                $builder->where('business_jobs.expires_at <=', date('Y-m-d H:i:s'));
                break;
            case 'all':
                // No additional filters
                break;
        }

        // Apply search
        if ($search) {
            $builder->groupStart()
                   ->like('business_jobs.title', $search)
                   ->orLike('business_jobs.description', $search)
                   ->orLike('store.name', $search)
                   ->orLike('store.city', $search)
                   ->groupEnd();
        }

        // Apply employment type filter
        if ($employmentType) {
            $builder->where('business_jobs.employment_type', $employmentType);
        }

        // Apply business filter
        if ($business) {
            $builder->where('business_jobs.store_id', $business);
        }

        // Get total count for pagination
        $totalJobs = $builder->countAllResults(false);

        // Get paginated results
        $jobs = $builder->orderBy('business_jobs.created_at', 'DESC')
                       ->limit($perPage, ($page - 1) * $perPage)
                       ->findAll();

        // Format jobs for display
        foreach ($jobs as &$job) {
            $job = $this->jobModel->formatJobDisplay($job);
        }

        // Get summary statistics
        $stats = $this->getJobStats();

        // Get businesses for filter dropdown
        $businesses = $this->storeModel->select('store_id, name, city, state')
                                      ->where('status', 1)
                                      ->orderBy('name')
                                      ->findAll();

        // Calculate pagination
        $totalPages = ceil($totalJobs / $perPage);

        $data = [
            'title' => 'Job Postings Management',
            'jobs' => $jobs,
            'stats' => $stats,
            'businesses' => $businesses,
            'filters' => [
                'current_filter' => $filter,
                'search' => $search,
                'employment_type' => $employmentType,
                'business' => $business
            ],
            'pagination' => [
                'current_page' => $page,
                'total_pages' => $totalPages,
                'total_jobs' => $totalJobs,
                'per_page' => $perPage
            ],
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('admin/jobs/index', $data);
    }

    /**
     * View job details
     */
    public function view($id)
    {
        if (!$this->isAdminLoggedIn()) {
            return redirect()->to('/admin/auth/login');
        }

        $job = $this->jobModel->select('business_jobs.*, store.name as store_name, store.city, store.state, store.phone, store.email, store.address, business_users.first_name, business_users.last_name, business_users.email as user_email')
                             ->join('store', 'store.store_id = business_jobs.store_id')
                             ->join('business_users', 'business_users.id = business_jobs.created_by', 'left')
                             ->find($id);

        if (!$job) {
            return redirect()->to('/admin/jobs')->with('error', 'Job posting not found.');
        }

        $job = $this->jobModel->formatJobDisplay($job);

        $data = [
            'title' => 'Job Details - ' . $job['title'],
            'job' => $job
        ];

        return view('admin/jobs/view', $data);
    }

    /**
     * Delete job posting
     */
    public function delete($id)
    {
        if (!$this->isAdminLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authenticated']);
        }

        $job = $this->jobModel->select('business_jobs.*, store.name as store_name, business_users.email as user_email')
                             ->join('store', 'store.store_id = business_jobs.store_id')
                             ->join('business_users', 'business_users.id = business_jobs.created_by', 'left')
                             ->find($id);

        if (!$job) {
            return $this->response->setJSON(['success' => false, 'message' => 'Job posting not found']);
        }

        // Log admin action
        log_message('info', 'Admin deleted job posting: ID ' . $id . ' - ' . $job['title'] . ' from ' . $job['store_name'] . ' by admin ' . $this->session->get('admin_name'));

        if ($this->jobModel->delete($id)) {
            // Send notification to business user
            $this->notifyBusinessJobDeleted($job);
            
            return $this->response->setJSON(['success' => true, 'message' => 'Job posting deleted successfully']);
        }

        return $this->response->setJSON(['success' => false, 'message' => 'Failed to delete job posting']);
    }

    /**
     * Get job statistics for dashboard
     */
    private function getJobStats()
    {
        $db = \Config\Database::connect();
        
        // Total jobs
        $totalJobs = $this->jobModel->countAllResults(false);
        
        // Active jobs
        $activeJobs = $this->jobModel->where('is_active', 1)
                                    ->where('expires_at >', date('Y-m-d H:i:s'))
                                    ->countAllResults(false);
        
        // Jobs expiring within 7 days
        $expiringJobs = $this->jobModel->where('is_active', 1)
                                      ->where('expires_at <=', date('Y-m-d H:i:s', strtotime('+7 days')))
                                      ->where('expires_at >', date('Y-m-d H:i:s'))
                                      ->countAllResults(false);
        
        // Expired jobs
        $expiredJobs = $this->jobModel->where('expires_at <=', date('Y-m-d H:i:s'))
                                     ->countAllResults(false);
        
        // Jobs by employment type
        $jobsByType = $db->query("
            SELECT employment_type, COUNT(*) as count 
            FROM business_jobs 
            WHERE is_active = 1 AND expires_at > NOW() 
            GROUP BY employment_type
        ")->getResultArray();
        
        // Recent job postings (last 7 days)
        $recentJobs = $this->jobModel->where('created_at >=', date('Y-m-d H:i:s', strtotime('-7 days')))
                                    ->countAllResults(false);
        
        // Businesses with active jobs
        $businessesWithJobs = $db->query("
            SELECT COUNT(DISTINCT store_id) as count 
            FROM business_jobs 
            WHERE is_active = 1 AND expires_at > NOW()
        ")->getRow()->count;

        return [
            'total_jobs' => $totalJobs,
            'active_jobs' => $activeJobs,
            'expiring_jobs' => $expiringJobs,
            'expired_jobs' => $expiredJobs,
            'recent_jobs' => $recentJobs,
            'businesses_with_jobs' => $businessesWithJobs,
            'jobs_by_type' => $jobsByType
        ];
    }

    /**
     * Export jobs data
     */
    public function export()
    {
        if (!$this->isAdminLoggedIn()) {
            return redirect()->to('/admin/auth/login');
        }

        $filter = $this->request->getGet('filter') ?? 'active';
        
        // Get jobs based on filter
        $builder = $this->jobModel->select('business_jobs.*, store.name as store_name, store.city, store.state, store.phone, store.email')
                                 ->join('store', 'store.store_id = business_jobs.store_id');

        switch ($filter) {
            case 'active':
                $builder->where('business_jobs.is_active', 1)
                       ->where('business_jobs.expires_at >', date('Y-m-d H:i:s'));
                break;
            case 'expiring':
                $expiringDate = date('Y-m-d H:i:s', strtotime('+7 days'));
                $builder->where('business_jobs.is_active', 1)
                       ->where('business_jobs.expires_at <=', $expiringDate)
                       ->where('business_jobs.expires_at >', date('Y-m-d H:i:s'));
                break;
        }

        $jobs = $builder->orderBy('business_jobs.created_at', 'DESC')->findAll();

        // Format for CSV export
        $csvData = [];
        $csvData[] = ['Business Name', 'Job Title', 'Employment Type', 'Salary Min', 'Salary Max', 'Salary Type', 'Contact Method', 'City', 'State', 'Posted Date', 'Expires Date', 'Status'];

        foreach ($jobs as $job) {
            $formatted = $this->jobModel->formatJobDisplay($job);
            $csvData[] = [
                $job['store_name'],
                $job['title'],
                $formatted['employment_type_display'],
                $job['salary_min'],
                $job['salary_max'] ?: '',
                ucfirst($job['salary_type']),
                ucfirst(str_replace('_', ' ', $job['contact_method'])),
                $job['city'],
                $job['state'],
                date('Y-m-d', strtotime($job['created_at'])),
                date('Y-m-d', strtotime($job['expires_at'])),
                $job['is_active'] ? 'Active' : 'Inactive'
            ];
        }

        // Generate CSV
        $filename = 'job_postings_' . $filter . '_' . date('Y-m-d') . '.csv';
        
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        foreach ($csvData as $row) {
            fputcsv($output, $row);
        }
        fclose($output);
        exit;
    }

    /**
     * Send notification to business about deleted job
     */
    private function notifyBusinessJobDeleted($job)
    {
        if (empty($job['user_email'])) {
            return false;
        }

        try {
            $email = \Config\Services::email();
            
            $subject = 'Job Posting Removed - ' . $job['title'];
            
            $message = "
Dear {$job['first_name']} {$job['last_name']},

Your job posting has been removed from our platform:

Job Title: {$job['title']}
Business: {$job['store_name']}
Removal Date: " . date('M j, Y g:i A') . "

This action was taken by our administrative team. If you believe this was done in error or have questions about our job posting policies, please contact us.

You can create a new job posting at any time through your business portal.

Best regards,
ActiveParkways/HOST Team
            ";

            $email->setTo($job['user_email']);
            $email->setSubject($subject);
            $email->setMessage($message);
            
            return $email->send();

        } catch (\Exception $e) {
            log_message('error', 'Failed to send job deletion notification: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if admin is logged in
     */
    private function isAdminLoggedIn()
    {
        return $this->session->get('admin_logged_in') === true && 
               !empty($this->session->get('admin_id'));
    }
}