<?php

namespace App\Controllers\Business;

use App\Controllers\BaseController;
use App\Models\BusinessUserModel;
use CodeIgniter\HTTP\ResponseInterface;

class Auth extends BaseController
{
    protected $businessUserModel;
    protected $session;

    public function __construct()
    {
        $this->businessUserModel = new BusinessUserModel();
        $this->session = \Config\Services::session();
        helper(['form']); // Add this line to load form helper
    }

    /**
     * Display login form
     */
    public function login()
    {
        // If already logged in, redirect to dashboard
        if ($this->isLoggedIn()) {
            return redirect()->to('/business/dashboard');
        }

        $data = [
            'title' => 'Business Portal Login',
            'error' => session()->getFlashdata('error'),
            'success' => session()->getFlashdata('success')
        ];

        return view('business/auth/login', $data);
    }

    /**
     * Process login attempt
     */
    public function attemptLogin()
    {
        $rules = [
            'username' => 'required|min_length[3]',
            'password' => 'required|min_length[6]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('error', 'Please fill in all required fields.');
        }

        $username = $this->request->getPost('username');
        $password = $this->request->getPost('password');

        // Find user by username
        $user = $this->businessUserModel->where('username', $username)
                                       ->where('status', 1)
                                       ->first();

        if (!$user || !password_verify($password, $user['password'])) {
            return redirect()->back()->withInput()->with('error', 'Invalid username or password.');
        }

        // Update last login
        $this->businessUserModel->update($user['id'], ['last_login' => date('Y-m-d H:i:s')]);

        // Set session data
        $sessionData = [
            'business_user_id' => $user['id'],
            'business_username' => $user['username'],
            'business_store_id' => $user['store_id'],
            'business_role' => $user['role'],
            'business_logged_in' => true
        ];
        $this->session->set($sessionData);

        // Check if this is first login (missing contact info)
        if (empty($user['first_name']) || empty($user['last_name']) || empty($user['phone'])) {
            return redirect()->to('/business/auth/first-login')->with('success', 'Welcome! Please complete your contact information.');
        }

        return redirect()->to('/business/dashboard')->with('success', 'Welcome back, ' . $user['first_name'] . '!');
    }

    /**
     * First login - collect contact information
     */
    public function firstLogin()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $userId = $this->session->get('business_user_id');
        $user = $this->businessUserModel->find($userId);

        // If contact info already complete, redirect to dashboard
        if (!empty($user['first_name']) && !empty($user['last_name']) && !empty($user['phone'])) {
            return redirect()->to('/business/dashboard');
        }

        $data = [
            'title' => 'Complete Your Profile',
            'user' => $user,
            'error' => session()->getFlashdata('error'),
            'success' => session()->getFlashdata('success')
        ];

        return view('business/auth/first_login', $data);
    }

    /**
     * Process first login contact info
     */
    public function saveContactInfo()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $rules = [
            'first_name' => 'required|min_length[2]|max_length[100]',
            'last_name' => 'required|min_length[2]|max_length[100]',
            'email' => 'required|valid_email|max_length[255]',
            'phone' => 'required|min_length[10]|max_length[20]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('error', 'Please fill in all required fields correctly.');
        }

        $userId = $this->session->get('business_user_id');
        
        $updateData = [
            'first_name' => $this->request->getPost('first_name'),
            'last_name' => $this->request->getPost('last_name'),
            'email' => $this->request->getPost('email'),
            'phone' => $this->request->getPost('phone')
        ];

        // Check if email is already used by another business user
        $existingUser = $this->businessUserModel->where('email', $updateData['email'])
                                               ->where('id !=', $userId)
                                               ->first();

        if ($existingUser) {
            return redirect()->back()->withInput()->with('error', 'This email address is already in use.');
        }

        if ($this->businessUserModel->update($userId, $updateData)) {
            return redirect()->to('/business/dashboard')->with('success', 'Profile completed successfully! Welcome to the Business Portal.');
        } else {
            return redirect()->back()->withInput()->with('error', 'Failed to save contact information. Please try again.');
        }
    }

    /**
     * Change password form
     */
    public function changePassword()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $data = [
            'title' => 'Change Password',
            'error' => session()->getFlashdata('error'),
            'success' => session()->getFlashdata('success')
        ];

        return view('business/auth/change_password', $data);
    }

    /**
     * Process password change
     */
    public function updatePassword()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $rules = [
            'current_password' => 'required',
            'new_password' => 'required|min_length[6]',
            'confirm_password' => 'required|matches[new_password]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->with('error', 'Please fill in all fields correctly. New password must be at least 6 characters and passwords must match.');
        }

        $userId = $this->session->get('business_user_id');
        $user = $this->businessUserModel->find($userId);

        // Verify current password
        if (!password_verify($this->request->getPost('current_password'), $user['password'])) {
            return redirect()->back()->with('error', 'Current password is incorrect.');
        }

        // Update password
        $newPassword = password_hash($this->request->getPost('new_password'), PASSWORD_DEFAULT);
        
        if ($this->businessUserModel->update($userId, ['password' => $newPassword])) {
            return redirect()->back()->with('success', 'Password changed successfully.');
        } else {
            return redirect()->back()->with('error', 'Failed to change password. Please try again.');
        }
    }

    /**
     * Logout
     */
    public function logout()
    {
        $this->session->remove(['business_user_id', 'business_username', 'business_store_id', 'business_role', 'business_logged_in']);
        return redirect()->to('/business/auth/login')->with('success', 'You have been logged out successfully.');
    }

    /**
     * Check if business user is logged in
     */
    private function isLoggedIn()
    {
        return $this->session->get('business_logged_in') === true && !empty($this->session->get('business_user_id'));
    }

    /**
     * Password recovery form
     */
    public function forgotPassword()
    {
        $data = [
            'title' => 'Forgot Password',
            'error' => session()->getFlashdata('error'),
            'success' => session()->getFlashdata('success')
        ];

        return view('business/auth/forgot_password', $data);
    }

    /**
     * Process password recovery request
     */
    public function processForgotPassword()
    {
        $rules = [
            'username' => 'required|min_length[3]'
        ];

        if (!$this->validate($rules)) {
            return redirect()->back()->withInput()->with('error', 'Please enter your username.');
        }

        $username = $this->request->getPost('username');
        $user = $this->businessUserModel->where('username', $username)->where('status', 1)->first();

        if (!$user) {
            // Don't reveal if username exists or not for security
            return redirect()->back()->with('success', 'If your username exists, you will receive password reset instructions.');
        }

        // Generate reset token
        $token = bin2hex(random_bytes(32));
        $this->businessUserModel->update($user['id'], ['verification_token' => $token]);

        // Here you would send an email with reset link
        // For now, just show success message
        return redirect()->back()->with('success', 'Password reset instructions have been sent. Please contact support for assistance.');
    }
}