<?php

namespace App\Controllers\Business;

use App\Controllers\BaseController;
use App\Models\BusinessUserModel;
use App\Models\StoreModel;

class Feedback extends BaseController
{
    protected $businessUserModel;
    protected $storeModel;
    protected $db;
    protected $session;

    public function __construct()
    {
        $this->businessUserModel = new BusinessUserModel();
        $this->storeModel = new StoreModel();
        $this->db = \Config\Database::connect();
        $this->session = \Config\Services::session();
    }

    /**
     * Main feedback dashboard
     */
    public function index()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $store = $this->storeModel->getStoreById($storeId);
        
        if (!$store) {
            return redirect()->to('/business/dashboard')->with('error', 'Store not found.');
        }

        // Get dashboard data
        $stats = $this->getFeedbackStats($storeId);
        $averages = $this->getAverageRatings($storeId);
        $recentFeedback = $this->getRecentFeedback($storeId, 10);
        $activeCases = $this->getActiveCases($storeId);
        $caseStats = $this->getCaseStats($storeId);
        $settings = $this->getFeedbackSettings($storeId);
        $qrCode = $this->getQRCode($storeId);

        $data = [
            'title' => 'Customer Feedback Dashboard',
            'store' => $store,
            'stats' => $stats,
            'averages' => $averages,
            'recent_feedback' => $recentFeedback,
            'active_cases' => $activeCases,
            'case_stats' => $caseStats,
            'settings' => $settings,
            'qr_code' => $qrCode,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/feedback/index', $data);
    }

    /**
     * Feedback settings page
     */
    public function settings()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $store = $this->storeModel->getStoreById($storeId);
        
        if (!$store) {
            return redirect()->to('/business/dashboard')->with('error', 'Store not found.');
        }

        $settings = $this->getFeedbackSettings($storeId);
        $qrCode = $this->getQRCode($storeId);

        $data = [
            'title' => 'Feedback Settings',
            'store' => $store,
            'settings' => $settings,
            'qr_code' => $qrCode,
            'qr_url' => $this->generateQRUrl($qrCode),
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/feedback/settings', $data);
    }

    /**
     * Update feedback settings
     */
    public function updateSettings()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        
        $settings = [
            'feedback_enabled' => $this->request->getPost('feedback_enabled') ? 1 : 0,
            'email_notifications' => $this->request->getPost('email_notifications') ? 1 : 0,
            'google_review_url' => trim($this->request->getPost('google_review_url')) ?: null,
            'google_redirect_threshold' => (int)$this->request->getPost('google_redirect_threshold') ?: 7,
            'resolution_threshold' => (int)$this->request->getPost('resolution_threshold') ?: 6,
            'require_contact_for_cases' => $this->request->getPost('require_contact_for_cases') ? 1 : 0,
            'auto_close_resolved' => $this->request->getPost('auto_close_resolved') ? 1 : 0,
            'response_time_goal' => (int)$this->request->getPost('response_time_goal') ?: 24
        ];

        // Validate Google review URL
        if ($settings['google_review_url'] && !filter_var($settings['google_review_url'], FILTER_VALIDATE_URL)) {
            return redirect()->back()->withInput()->with('error', 'Please enter a valid review URL');
        }

        try {
            $this->db->query("
                UPDATE store SET 
                    feedback_enabled = ?,
                    email_notifications = ?,
                    google_review_url = ?,
                    google_redirect_threshold = ?,
                    resolution_threshold = ?,
                    require_contact_for_cases = ?,
                    auto_close_resolved = ?,
                    response_time_goal = ?
                WHERE store_id = ?
            ", array_merge(array_values($settings), [$storeId]));

            return redirect()->to('/business/feedback/settings')->with('success', 'Settings updated successfully.');

        } catch (\Exception $e) {
            log_message('error', 'Failed to update feedback settings: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to update settings');
        }
    }

    /**
     * List all cases with pagination
     */
    public function cases()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $store = $this->storeModel->getStoreById($storeId);
        
        if (!$store) {
            return redirect()->to('/business/dashboard')->with('error', 'Store not found.');
        }

        $page = max((int)($this->request->getGet('page') ?? 1), 1);
        $limit = 10;
        $offset = ($page - 1) * $limit;

        $cases = $this->getCasesWithPagination($storeId, $limit, $offset);
        $totalCases = $this->getTotalCasesCount($storeId);
        
        $data = [
            'title' => 'Customer Feedback Cases',
            'store' => $store,
            'cases' => $cases,
            'pagination' => [
                'current_page' => $page,
                'total_pages' => ceil($totalCases / $limit),
                'total_items' => $totalCases,
                'per_page' => $limit
            ],
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/feedback/cases', $data);
    }

    /**
     * View individual case details
     */
    public function viewCase($caseId)
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        
        $case = $this->getCaseDetails($caseId, $storeId);
        
        if (!$case) {
            return redirect()->to('/business/feedback')->with('error', 'Case not found.');
        }

        $messages = $this->getCaseMessages($caseId);
        $unreadCount = $this->getUnreadMessageCount($caseId, 'business');
        
        // Mark messages as read
        $this->markMessagesAsRead($caseId, 'business');

        $data = [
            'title' => 'Case ' . $case['case_number'],
            'case' => $case,
            'messages' => $messages,
            'unread_count' => $unreadCount,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/feedback/case', $data);
    }

    /**
     * Send message to customer
     */
    public function sendMessage($caseId)
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $message = trim($this->request->getPost('message'));
        $senderId = $this->session->get('business_user_id');
        $storeId = $this->session->get('business_store_id');

        // Validation
        if (!$message || strlen($message) > 2000) {
            return redirect()->back()->with('error', 'Please enter a valid message.');
        }

        // Verify case belongs to this business
        $case = $this->getCaseDetails($caseId, $storeId);
        if (!$case) {
            return redirect()->to('/business/feedback')->with('error', 'Case not found.');
        }

        if ($case['status'] === 'closed') {
            return redirect()->back()->with('error', 'Cannot send message to closed case.');
        }

        try {
            $messageId = $this->generateUniqueId('msg_');
            
            // Insert message
            $this->db->query("
                INSERT INTO case_messages (message_id, case_id, sender_type, sender_id, message)
                VALUES (?, ?, 'business', ?, ?)
            ", [$messageId, $caseId, $senderId, $message]);

            // Update case status if needed
            $markInProgress = $this->request->getPost('mark_in_progress');
            if ($markInProgress && $case['status'] === 'open') {
                $this->updateCaseStatusInternal($caseId, 'in_progress');
            }

            // Update case timestamp
            $this->db->query("
                UPDATE feedback_cases 
                SET updated_at = NOW()
                WHERE case_id = ?
            ", [$caseId]);

            // Send email notification to customer
            $this->notifyCustomerOfMessage($case, $message);

            return redirect()->back()->with('success', 'Message sent successfully!');

        } catch (\Exception $e) {
            log_message('error', 'Failed to send case message: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to send message.');
        }
    }

    /**
     * Close a case
     */
    public function closeCase($caseId)
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        
        $case = $this->getCaseDetails($caseId, $storeId);
        if (!$case) {
            return redirect()->to('/business/feedback')->with('error', 'Case not found.');
        }

        try {
            $this->db->query("
                UPDATE feedback_cases 
                SET status = 'closed', closed_at = NOW()
                WHERE case_id = ?
            ", [$caseId]);

            // Add closure message
            $this->db->query("
                INSERT INTO case_messages (message_id, case_id, sender_type, message)
                VALUES (?, ?, 'system', 'Case closed by business')
            ", [$this->generateUniqueId('sys_'), $caseId]);

            // Notify customer
            $this->notifyCustomerOfClosure($case);

            return redirect()->to('/business/feedback')->with('success', 'Case closed successfully.');

        } catch (\Exception $e) {
            log_message('error', 'Failed to close case: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to close case.');
        }
    }

    /**
     * Update case priority (disabled until priority column is added to database)
     */
    public function updatePriority($caseId)
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        // Priority feature not yet implemented in database
        return redirect()->back()->with('error', 'Priority feature is not yet available.');
        
        /* 
        // Uncomment when priority column is added to feedback_cases table:
        $priority = $this->request->getPost('priority');
        $storeId = $this->session->get('business_store_id');
        
        if (!in_array($priority, ['normal', 'urgent'])) {
            return redirect()->back()->with('error', 'Invalid priority level.');
        }

        $case = $this->getCaseDetails($caseId, $storeId);
        if (!$case) {
            return redirect()->to('/business/feedback')->with('error', 'Case not found.');
        }

        try {
            $this->db->query("
                UPDATE feedback_cases 
                SET priority = ?, updated_at = NOW()
                WHERE case_id = ?
            ", [$priority, $caseId]);

            return redirect()->back()->with('success', 'Priority updated successfully.');

        } catch (\Exception $e) {
            log_message('error', 'Failed to update priority: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to update priority.');
        }
        */
    }

    /**
     * Messaging interface
     */
    public function messages($caseId)
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        
        $case = $this->db->query("
            SELECT 
                fc.*,
                cf.customer_name,
                DATE_FORMAT(fc.created_at, '%M %d, %Y at %l:%i %p') as created_at_formatted
            FROM feedback_cases fc
            JOIN customer_feedback cf ON fc.feedback_id = cf.feedback_id
            WHERE fc.case_id = ? AND fc.store_id = ?
        ", [$caseId, $storeId])->getRowArray();

        if (!$case) {
            return redirect()->to('/business/feedback')->with('error', 'Case not found.');
        }

        $messages = $this->getCaseMessages($caseId);
        $unreadCount = $this->getUnreadMessageCount($caseId, 'business');
        
        // Mark messages as read
        $this->markMessagesAsRead($caseId, 'business');

        $data = [
            'title' => 'Messages - Case ' . $case['case_number'],
            'case' => $case,
            'messages' => $messages,
            'unread_count' => $unreadCount
        ];

        return view('business/feedback/messages', $data);
    }

    /**
     * AJAX: Get case details for modal
     */
    public function caseDetails($caseId)
    {
        if (!$this->isLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Unauthorized']);
        }

        $storeId = $this->session->get('business_store_id');
        
        try {
            $case = $this->getCaseDetails($caseId, $storeId);

            if (!$case) {
                return $this->response->setJSON(['success' => false, 'message' => 'Case not found']);
            }

            $recentMessages = $this->db->query("
                SELECT 
                    message,
                    sender_type,
                    DATE_FORMAT(created_at, '%M %d at %l:%i %p') as created_at_formatted
                FROM case_messages 
                WHERE case_id = ? 
                ORDER BY created_at DESC 
                LIMIT 3
            ", [$caseId])->getResultArray();

            $recentMessages = array_reverse($recentMessages);

            $data = [
                'case' => $case,
                'recent_messages' => $recentMessages
            ];

            return $this->response->setJSON([
                'success' => true,
                'html' => view('business/feedback/case_modal', $data)
            ]);

        } catch (\Exception $e) {
            log_message('error', 'Case details AJAX error: ' . $e->getMessage());
            return $this->response->setJSON(['success' => false, 'message' => 'Unable to load case details']);
        }
    }

    /**
     * AJAX: Poll for new messages
     */
    public function pollMessages($caseId)
    {
        if (!$this->isLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Unauthorized']);
        }

        $storeId = $this->session->get('business_store_id');
        $lastId = $this->request->getGet('last_id') ?? 0;

        try {
            $case = $this->db->query("
                SELECT case_id FROM feedback_cases 
                WHERE case_id = ? AND store_id = ?
            ", [$caseId, $storeId])->getRowArray();

            if (!$case) {
                return $this->response->setJSON(['success' => false, 'message' => 'Case not found']);
            }

            $messages = $this->db->query("
                SELECT 
                    message_id,
                    message,
                    sender_type,
                    DATE_FORMAT(created_at, '%M %d at %l:%i %p') as created_at_formatted
                FROM case_messages 
                WHERE case_id = ? AND message_id > ?
                ORDER BY created_at ASC
            ", [$caseId, $lastId])->getResultArray();

            return $this->response->setJSON([
                'success' => true,
                'messages' => $messages
            ]);

        } catch (\Exception $e) {
            log_message('error', 'Poll messages error: ' . $e->getMessage());
            return $this->response->setJSON(['success' => false, 'message' => 'Failed to poll messages']);
        }
    }

    /**
     * AJAX: Mark messages as read
     */
    public function markMessagesRead($caseId)
    {
        if (!$this->isLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Unauthorized']);
        }

        try {
            $this->markMessagesAsRead($caseId, 'business');
            return $this->response->setJSON(['success' => true]);
        } catch (\Exception $e) {
            log_message('error', 'Mark read error: ' . $e->getMessage());
            return $this->response->setJSON(['success' => false, 'message' => 'Failed to mark as read']);
        }
    }

    // ==========================================
    // PRIVATE HELPER METHODS
    // ==========================================

    /**
     * Get feedback statistics
     */
    private function getFeedbackStats($storeId)
    {
        $stats = $this->db->query("
            SELECT 
                COUNT(*) as total_feedback,
                AVG(overall_satisfaction) as avg_satisfaction,
                AVG(net_promoter_score) as avg_nps,
                COUNT(CASE WHEN overall_satisfaction >= 7 AND net_promoter_score >= 7 THEN 1 END) as positive_feedback,
                COUNT(CASE WHEN overall_satisfaction <= 6 OR net_promoter_score <= 6 THEN 1 END) as negative_feedback,
                COUNT(CASE WHEN google_redirect_offered = 1 THEN 1 END) as review_redirects,
                COUNT(CASE WHEN google_redirect_accepted = 1 THEN 1 END) as review_accepts,
                COUNT(CASE WHEN google_redirect_accepted = 1 THEN 1 END) as google_redirects
            FROM customer_feedback 
            WHERE store_id = ? AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        ", [$storeId])->getRowArray();

        // Get case statistics
        $caseStats = $this->db->query("
            SELECT 
                COUNT(CASE WHEN status IN ('open', 'in_progress') THEN 1 END) as open_cases,
                COUNT(CASE WHEN status = 'resolved' THEN 1 END) as resolved_cases,
                COUNT(CASE WHEN status = 'closed' THEN 1 END) as closed_cases
            FROM feedback_cases 
            WHERE store_id = ?
        ", [$storeId])->getRowArray();

        // Calculate response rate
        $totalCases = $this->db->query("SELECT COUNT(*) as count FROM feedback_cases WHERE store_id = ?", [$storeId])->getRowArray();
        $respondedCases = $this->db->query("
            SELECT COUNT(DISTINCT fc.case_id) as count 
            FROM feedback_cases fc
            JOIN case_messages cm ON fc.case_id = cm.case_id
            WHERE fc.store_id = ? AND cm.sender_type = 'business'
        ", [$storeId])->getRowArray();

        $responseRate = $totalCases['count'] > 0 ? round(($respondedCases['count'] / $totalCases['count']) * 100) : 0;

        // Merge all results
        $allStats = array_merge($stats, $caseStats);
        $allStats['response_rate'] = $responseRate;

        return $allStats;
    }

    /**
     * Get average ratings
     */
    private function getAverageRatings($storeId)
    {
        return $this->db->query("
            SELECT 
                COALESCE(AVG(overall_satisfaction), 0) as satisfaction,
                COALESCE(AVG(net_promoter_score), 0) as nps
            FROM customer_feedback 
            WHERE store_id = ?
        ", [$storeId])->getRowArray();
    }

    /**
     * Get recent feedback
     */
    private function getRecentFeedback($storeId, $limit)
    {
        return $this->db->query("
            SELECT 
                feedback_id,
                overall_satisfaction,
                net_promoter_score,
                comment,
                customer_name,
                google_redirect_offered,
                google_redirect_accepted,
                DATE_FORMAT(created_at, '%M %d at %l:%i %p') as created_at_formatted,
                created_at
            FROM customer_feedback 
            WHERE store_id = ?
            ORDER BY created_at DESC
            LIMIT ?
        ", [$storeId, $limit])->getResultArray();
    }

    /**
     * Get active cases
     */
    private function getActiveCases($storeId)
    {
        return $this->db->query("
            SELECT 
                fc.case_id,
                fc.case_number,
                fc.status,
                fc.created_at,
                fc.updated_at,
                fc.closed_at,
                cf.overall_satisfaction as satisfaction_score,
                cf.net_promoter_score as nps_score,
                cf.comment as feedback_comment,
                cf.customer_name,
                cf.customer_contact,
                DATE_FORMAT(fc.created_at, '%M %d, %Y at %l:%i %p') as created_at_formatted,
                DATEDIFF(NOW(), fc.created_at) as days_since_created,
                (SELECT COUNT(*) FROM case_messages WHERE case_id = fc.case_id AND sender_type = 'customer' AND read_at IS NULL) as unread_messages
            FROM feedback_cases fc
            JOIN customer_feedback cf ON fc.feedback_id = cf.feedback_id
            WHERE fc.store_id = ? AND fc.status IN ('open', 'in_progress')
            ORDER BY fc.created_at DESC
        ", [$storeId])->getResultArray();
    }

    /**
     * Get case statistics
     */
    private function getCaseStats($storeId)
    {
        return $this->db->query("
            SELECT 
                COUNT(*) as total_cases,
                COUNT(CASE WHEN status = 'open' THEN 1 END) as open_cases,
                COUNT(CASE WHEN status = 'in_progress' THEN 1 END) as in_progress_cases,
                COUNT(CASE WHEN status = 'resolved' THEN 1 END) as resolved_cases,
                COUNT(CASE WHEN status = 'closed' THEN 1 END) as closed_cases
            FROM feedback_cases 
            WHERE store_id = ?
        ", [$storeId])->getRowArray();
    }

    /**
     * Get feedback settings
     */
    private function getFeedbackSettings($storeId)
    {
        $store = $this->db->query("
            SELECT 
                feedback_enabled,
                email_notifications,
                google_review_url,
                google_redirect_threshold,
                resolution_threshold,
                require_contact_for_cases,
                auto_close_resolved,
                response_time_goal
            FROM store 
            WHERE store_id = ?
        ", [$storeId])->getRowArray();

        return [
            'feedback_enabled' => $store['feedback_enabled'] ?? 1,
            'email_notifications' => $store['email_notifications'] ?? 1,
            'google_review_url' => $store['google_review_url'] ?? '',
            'google_redirect_threshold' => $store['google_redirect_threshold'] ?? 7,
            'resolution_threshold' => $store['resolution_threshold'] ?? 6,
            'require_contact_for_cases' => $store['require_contact_for_cases'] ?? 1,
            'auto_close_resolved' => $store['auto_close_resolved'] ?? 0,
            'response_time_goal' => $store['response_time_goal'] ?? 24
        ];
    }

    /**
     * Get QR code
     */
    private function getQRCode($storeId)
    {
        $result = $this->db->query("SELECT feedback_qr_code FROM store WHERE store_id = ?", [$storeId])->getRowArray();
        return $result['feedback_qr_code'] ?? 'qr_' . $storeId;
    }

    /**
     * Get cases with pagination
     */
    private function getCasesWithPagination($storeId, $limit, $offset)
    {
        return $this->db->query("
            SELECT 
                fc.*,
                cf.customer_name,
                cf.overall_satisfaction,
                cf.net_promoter_score,
                cf.comment,
                DATE_FORMAT(fc.created_at, '%M %d, %Y at %l:%i %p') as created_at_formatted
            FROM feedback_cases fc
            JOIN customer_feedback cf ON fc.feedback_id = cf.feedback_id
            WHERE fc.store_id = ?
            ORDER BY fc.created_at DESC
            LIMIT ? OFFSET ?
        ", [$storeId, $limit, $offset])->getResultArray();
    }

    /**
     * Get total cases count
     */
    private function getTotalCasesCount($storeId)
    {
        $result = $this->db->query("SELECT COUNT(*) as count FROM feedback_cases WHERE store_id = ?", [$storeId])->getRowArray();
        return $result['count'] ?? 0;
    }

    /**
     * Get case details
     */
    private function getCaseDetails($caseId, $storeId)
    {
        return $this->db->query("
            SELECT 
                fc.*,
                cf.overall_satisfaction as satisfaction_score,
                cf.net_promoter_score as nps_score,
                cf.comment as feedback_comment,
                cf.customer_name,
                cf.customer_contact,
                DATE_FORMAT(fc.created_at, '%M %d, %Y at %l:%i %p') as created_at_formatted,
                DATE_FORMAT(fc.updated_at, '%M %d, %Y at %l:%i %p') as updated_at_formatted,
                DATE_FORMAT(fc.closed_at, '%M %d, %Y at %l:%i %p') as closed_at_formatted
            FROM feedback_cases fc
            JOIN customer_feedback cf ON fc.feedback_id = cf.feedback_id
            WHERE fc.case_id = ? AND fc.store_id = ?
        ", [$caseId, $storeId])->getRowArray();
    }

    /**
     * Get case messages
     */
    private function getCaseMessages($caseId)
    {
        return $this->db->query("
            SELECT 
                message_id,
                message,
                sender_type,
                created_at,
                read_at,
                DATE_FORMAT(created_at, '%M %d at %l:%i %p') as created_at_formatted
            FROM case_messages 
            WHERE case_id = ?
            ORDER BY created_at ASC
        ", [$caseId])->getResultArray();
    }

    /**
     * Get unread message count
     */
    private function getUnreadMessageCount($caseId, $forUser)
    {
        $condition = $forUser === 'business' ? "sender_type = 'customer'" : "sender_type = 'business'";
        
        $result = $this->db->query("
            SELECT COUNT(*) as count 
            FROM case_messages 
            WHERE case_id = ? AND {$condition} AND read_at IS NULL
        ", [$caseId])->getRowArray();

        return $result['count'] ?? 0;
    }

    /**
     * Mark messages as read
     */
    private function markMessagesAsRead($caseId, $forUser)
    {
        $condition = $forUser === 'business' ? "sender_type = 'customer'" : "sender_type = 'business'";
        
        $this->db->query("
            UPDATE case_messages 
            SET read_at = NOW() 
            WHERE case_id = ? AND {$condition} AND read_at IS NULL
        ", [$caseId]);
    }

    /**
     * Update case status
     */
    private function updateCaseStatusInternal($caseId, $status)
    {
        $this->db->query("
            UPDATE feedback_cases 
            SET status = ?, updated_at = NOW() 
            WHERE case_id = ?
        ", [$status, $caseId]);

        // Add system message
        $this->db->query("
            INSERT INTO case_messages (message_id, case_id, sender_type, message)
            VALUES (?, ?, 'system', ?)
        ", [
            $this->generateUniqueId('sys_'),
            $caseId,
            "Case status changed to: " . ucfirst(str_replace('_', ' ', $status))
        ]);
    }

    /**
     * Utility methods
     */
    private function generateQRUrl($qrCode)
    {
        return base_url('feedback/' . $qrCode);
    }

    private function generateUniqueId($prefix)
    {
        return $prefix . uniqid() . '_' . mt_rand(1000, 9999);
    }

    private function isLoggedIn()
    {
        return $this->session->get('business_logged_in') === true && !empty($this->session->get('business_user_id'));
    }

    private function notifyCustomerOfMessage($case, $message)
    {
        log_message('info', "Customer message notification needed for case {$case['case_number']}");
    }

    private function notifyCustomerOfClosure($case)
    {
        log_message('info', "Customer closure notification needed for case {$case['case_number']}");
    }
}