<?php

namespace App\Controllers\Business;

use App\Controllers\BaseController;
use App\Models\BusinessJobModel;
use App\Models\BusinessUserModel;
use App\Models\StoreModel;
use App\Services\BusinessEmailService;

class Jobs extends BaseController
{
    protected $jobModel;
    protected $businessUserModel;
    protected $storeModel;
    protected $emailService;
    protected $session;

    public function __construct()
    {
        $this->jobModel = new BusinessJobModel();
        $this->businessUserModel = new BusinessUserModel();
        $this->storeModel = new StoreModel();
        $this->emailService = new BusinessEmailService();
        $this->session = \Config\Services::session();
    }

    /**
     * Jobs management page with calendar view
     */
    public function index()
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $store = $this->storeModel->getStoreById($storeId);
        
        if (!$store) {
            return redirect()->to('/business/dashboard')->with('error', 'Store not found.');
        }

        // Get jobs for calendar display (3 months range)
        $startDate = date('Y-m-01', strtotime('-1 month'));
        $endDate = date('Y-m-t', strtotime('+2 months'));
        $jobs = $this->jobModel->getStoreJobs($storeId, $startDate, $endDate);

        // Format jobs for calendar
        $calendarEvents = [];
        foreach ($jobs as $job) {
            $formattedJob = $this->jobModel->formatJobDisplay($job);
            $calendarEvents[] = [
                'id' => $job['id'],
                'title' => $job['title'] . ' (' . ucfirst($job['employment_type']) . ')',
                'start' => $job['created_at'],
                'end' => $job['expires_at'] ? date('Y-m-d', strtotime($job['expires_at'] . ' +1 day')) : null,
                'backgroundColor' => $job['is_active'] ? '#10b981' : '#6b7280',
                'borderColor' => $job['is_active'] ? '#059669' : '#4b5563',
                'extendedProps' => [
                    'description' => $job['description'],
                    'employment_type' => $job['employment_type'],
                    'salary_display' => $formattedJob['salary_display'],
                    'contact_method' => $formattedJob['contact_method'],
                    'is_active' => $job['is_active'],
                    'expires_at' => $job['expires_at'],
                    'times_renewed' => $job['times_renewed'] ?? 0
                ]
            ];
        }

        // Get statistics
        $stats = $this->jobModel->getJobStats($storeId);
        
        // Get active jobs
        $activeJobs = $this->jobModel->getActiveJobs($storeId, 3);
        foreach ($activeJobs as &$job) {
            $job = $this->jobModel->formatJobDisplay($job);
        }

        // Get expiring jobs (within 5 days)
        $expiringJobs = $this->jobModel->getExpiringJobs($storeId, 5);

        $data = [
            'title' => 'Manage Job Postings',
            'store' => $store,
            'calendar_events' => json_encode($calendarEvents),
            'stats' => $stats,
            'active_jobs' => $activeJobs,
            'expiring_jobs' => $expiringJobs,
            'success' => session()->getFlashdata('success'),
            'error' => session()->getFlashdata('error')
        ];

        return view('business/jobs/index', $data);
    }

    /**
     * Create new job
     */
    public function create()
    {
        log_message('debug', 'Job CREATE: Method called, request method: ' . $this->request->getMethod());
        
        if (!$this->isLoggedIn()) {
            log_message('debug', 'Job CREATE: User not logged in, redirecting');
            return redirect()->to('/business/auth/login');
        }

        if ($this->request->getMethod() === 'post' || $this->request->getPost()) {
            log_message('debug', 'Job CREATE: POST request received, calling save()');
            return $this->save();
        }

        // Check if store can add more active jobs (3 job limit)
        $storeId = $this->session->get('business_store_id');
        $activeJobCount = $this->jobModel->getActiveJobCount($storeId);
        
        if ($activeJobCount >= 3) {
            return redirect()->to('/business/jobs')->with('error', 'You can only have 3 active job postings at a time. Please deactivate an existing job to add a new one.');
        }

        log_message('debug', 'Job CREATE: GET request - showing form');
        
        $data = [
            'title' => 'Post New Job',
            'job' => null,
            'is_edit' => false,
            'active_job_count' => $activeJobCount
        ];

        return view('business/jobs/form', $data);
    }

    /**
     * Edit existing job
     */
    public function edit($id)
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $job = $this->jobModel->where('store_id', $storeId)->find($id);

        if (!$job) {
            return redirect()->to('/business/jobs')->with('error', 'Job posting not found.');
        }

        if ($this->request->getMethod() === 'post' || $this->request->getPost()) {
            log_message('debug', 'Job EDIT: POST request received for ID ' . $id . ', calling save()');
            return $this->save($id);
        }

        $data = [
            'title' => 'Edit Job Posting',
            'job' => $job,
            'is_edit' => true
        ];

        return view('business/jobs/form', $data);
    }

    /**
     * Save job (create or update)
     */
    private function save($id = null)
    {
        log_message('debug', 'Job SAVE: Method started. ID: ' . ($id ?? 'null'));
        
        $storeId = $this->session->get('business_store_id');
        $userId = $this->session->get('business_user_id');
        
        log_message('debug', 'Job SAVE: Store ID: ' . $storeId . ', User ID: ' . $userId);
        log_message('debug', 'Job SAVE: POST data: ' . json_encode($this->request->getPost()));

        // Validation rules
        $rules = [
            'title' => 'required|min_length[3]|max_length[255]',
            'description' => 'required|min_length[10]',
            'employment_type' => 'required|in_list[full_time,part_time,contract,seasonal]',
            'salary_min' => 'required|numeric|greater_than[0]',
            'salary_type' => 'required|in_list[hourly,monthly,yearly]',
            'contact_method' => 'required|in_list[phone,email,walk_in,online]'
        ];

        // Conditional validation based on contact method
        $contactMethod = $this->request->getPost('contact_method');
        if ($contactMethod === 'online') {
            $rules['contact_value'] = 'required|valid_url';
        }

       log_message('debug', 'Job SAVE: Starting validation with rules: ' . json_encode($rules));

        if (!$this->validate($rules)) {
            $errors = $this->validator->getErrors();
            log_message('debug', 'Job SAVE: Validation FAILED. Errors: ' . json_encode($errors));
            return redirect()->back()->withInput()->with('validation', $errors);
        }

        log_message('debug', 'Job SAVE: Validation PASSED');

        // Check active job limit for new jobs
        if (!$id) {
            $activeJobCount = $this->jobModel->getActiveJobCount($storeId);
            if ($activeJobCount >= 3) {
                return redirect()->back()->withInput()->with('error', 'You can only have 3 active job postings at a time.');
            }
        }

        // Prepare data for insert/update
        $data = [
            'store_id' => (string)$storeId,
            'title' => trim($this->request->getPost('title')),
            'description' => trim($this->request->getPost('description')),
            'employment_type' => $this->request->getPost('employment_type'),
            'salary_min' => (float)$this->request->getPost('salary_min'),
            'salary_type' => $this->request->getPost('salary_type'),
            'contact_method' => $contactMethod,
            'is_active' => $this->request->getPost('is_active') ? 1 : 0
        ];

        // Set optional salary_max
        if ($this->request->getPost('salary_max')) {
            $data['salary_max'] = (float)$this->request->getPost('salary_max');
        } else {
            $data['salary_max'] = null;
        }

        // Set contact value for online applications
        if ($contactMethod === 'online') {
            $data['contact_value'] = $this->request->getPost('contact_value');
        } else {
            $data['contact_value'] = null;
        }

        // Set expiration date (30 days from now for new jobs)
        if (!$id) {
            $data['expires_at'] = date('Y-m-d H:i:s', strtotime('+30 days'));
            $data['created_by'] = (int)$userId;
            $data['times_renewed'] = 0;
            $data['renewal_email_sent'] = 0;
        } else {
            // For updates, don't change created_by or expiration unless specifically renewed
            unset($data['created_by']);
        }

        log_message('debug', 'Job SAVE: Data to save: ' . json_encode($data));

        // Save the job
        try {
            if ($id) {
                log_message('debug', 'Job SAVE: Attempting UPDATE for ID: ' . $id);
                $success = $this->jobModel->update($id, $data);
                $action = 'updated';
                log_message('debug', 'Job SAVE: Update result: ' . ($success ? 'SUCCESS' : 'FAILED'));
            } else {
                log_message('debug', 'Job SAVE: Attempting INSERT');
                $insertResult = $this->jobModel->insert($data);
                $success = $insertResult !== false;
                $insertId = $this->jobModel->getInsertID();
                $action = 'created';
                log_message('debug', 'Job SAVE: Insert result: ' . ($success ? 'SUCCESS' : 'FAILED') . ', Insert ID: ' . $insertId);
                
                if ($success) {
                    $id = $insertId;
                }
            }
            
        } catch (\Exception $e) {
            log_message('error', 'Job SAVE: Exception during save: ' . $e->getMessage());
            $success = false;
        }

        if ($success) {
            log_message('debug', 'Job SAVE: Save successful');
            
            // Notify admin of new job posting
            if ($action === 'created') {
                $this->notifyAdminJobChange($id, 'created');
            }
            
            return redirect()->to(base_url('business/jobs'))->with('success', "Job posting {$action} successfully!");
        } else {
            log_message('error', 'Job SAVE: Save FAILED - redirecting with error');
            return redirect()->back()->withInput()->with('error', "Failed to save job posting. Please try again.");
        }
    }

    /**
     * Toggle job active status
     */
    public function toggle($id)
    {
        if (!$this->isLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authenticated']);
        }

        $storeId = $this->session->get('business_store_id');
        $job = $this->jobModel->where('store_id', $storeId)->find($id);

        if (!$job) {
            return $this->response->setJSON(['success' => false, 'message' => 'Job posting not found']);
        }

        $newStatus = $job['is_active'] ? 0 : 1;
        
        // If activating, check active job limit
        if ($newStatus === 1) {
            $activeJobCount = $this->jobModel->getActiveJobCount($storeId);
            if ($activeJobCount >= 3) {
                return $this->response->setJSON([
                    'success' => false, 
                    'message' => 'You can only have 3 active job postings at a time.'
                ]);
            }
        }

        if ($this->jobModel->update($id, ['is_active' => $newStatus])) {
            // Notify admin
            $this->notifyAdminJobChange($id, $newStatus ? 'activated' : 'deactivated');
            
            return $this->response->setJSON([
                'success' => true, 
                'message' => 'Job posting ' . ($newStatus ? 'activated' : 'deactivated') . ' successfully!'
            ]);
        }

        return $this->response->setJSON(['success' => false, 'message' => 'Failed to update job status']);
    }

    /**
     * Renew job posting for another 30 days
     */
    public function renew($id)
    {
        if (!$this->isLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authenticated']);
        }

        $storeId = $this->session->get('business_store_id');
        $job = $this->jobModel->where('store_id', $storeId)->find($id);

        if (!$job) {
            return $this->response->setJSON(['success' => false, 'message' => 'Job posting not found']);
        }

        // Extend expiration by 30 days from current expiration or now (whichever is later)
        $currentExpiration = strtotime($job['expires_at']);
        $now = time();
        $newExpiration = date('Y-m-d H:i:s', strtotime('+30 days', max($currentExpiration, $now)));
        
        $updateData = [
            'expires_at' => $newExpiration,
            'times_renewed' => ($job['times_renewed'] ?? 0) + 1,
            'renewal_email_sent' => 0,
            'is_active' => 1 // Reactivate when renewed
        ];

        if ($this->jobModel->update($id, $updateData)) {
            // Notify admin
            $this->notifyAdminJobChange($id, 'renewed');
            
            return $this->response->setJSON([
                'success' => true, 
                'message' => 'Job posting renewed for 30 more days!',
                'new_expiration' => date('M j, Y', strtotime($newExpiration))
            ]);
        }

        return $this->response->setJSON(['success' => false, 'message' => 'Failed to renew job posting']);
    }

    /**
     * Delete job posting
     */
    public function delete($id)
    {
        if (!$this->isLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authenticated']);
        }

        $storeId = $this->session->get('business_store_id');
        $job = $this->jobModel->where('store_id', $storeId)->find($id);

        if (!$job) {
            return $this->response->setJSON(['success' => false, 'message' => 'Job posting not found']);
        }

        if ($this->jobModel->delete($id)) {
            // Notify admin
            $this->notifyAdminJobChange($id, 'deleted', $job);
            
            return $this->response->setJSON(['success' => true, 'message' => 'Job posting deleted successfully!']);
        }

        return $this->response->setJSON(['success' => false, 'message' => 'Failed to delete job posting']);
    }

    /**
     * Get job details via AJAX
     */
    public function details($id)
    {
        if (!$this->isLoggedIn()) {
            return $this->response->setJSON(['success' => false, 'message' => 'Not authenticated']);
        }

        $storeId = $this->session->get('business_store_id');
        $job = $this->jobModel->where('store_id', $storeId)->find($id);

        if (!$job) {
            return $this->response->setJSON(['success' => false, 'message' => 'Job posting not found']);
        }

        $job = $this->jobModel->formatJobDisplay($job);

        return $this->response->setJSON(['success' => true, 'job' => $job]);
    }

    /**
     * Preview job as it will appear to customers
     */
    public function preview($id)
    {
        if (!$this->isLoggedIn()) {
            return redirect()->to('/business/auth/login');
        }

        $storeId = $this->session->get('business_store_id');
        $job = $this->jobModel->where('store_id', $storeId)->find($id);

        if (!$job) {
            return redirect()->to('/business/jobs')->with('error', 'Job posting not found.');
        }

        $store = $this->storeModel->getStoreById($storeId);
        $job = $this->jobModel->formatJobDisplay($job);

        $data = [
            'title' => 'Job Posting Preview',
            'store' => $store,
            'job' => $job
        ];

        return view('business/jobs/preview', $data);
    }

    /**
     * Send email notification to admin about job changes
     */
    private function notifyAdminJobChange($jobId, $action, $jobData = null)
    {
        try {
            // Get job with creator info if not provided
            if (!$jobData) {
                $jobData = $this->jobModel->getJobWithCreator($jobId);
            }
            
            if (!$jobData) {
                return false;
            }

            // Get store info
            $store = $this->storeModel->getStoreById($jobData['store_id']);
            if (!$store) {
                return false;
            }

            // Get user info
            $user = $this->businessUserModel->find($this->session->get('business_user_id'));
            
            $subject = "Job posting {$action} - {$store['name']}";
            
            $data = [
                'action' => $action,
                'store' => $store,
                'job' => $jobData,
                'user' => $user,
                'admin_url' => base_url('admin/jobs'),
                'timestamp' => date('Y-m-d H:i:s')
            ];

            // Use existing email service pattern
            $this->emailService->notifyAdminJobChange($data);
            
            return true;

        } catch (\Exception $e) {
            log_message('error', "Failed to send job notification: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if business user is logged in
     */
    private function isLoggedIn()
    {
        return $this->session->get('business_logged_in') === true && 
               !empty($this->session->get('business_user_id'));
    }
}