<?php
// Cache-busting headers for development
header("Cache-Control: no-cache, no-store, must-revalidate");
header("Pragma: no-cache");
header("Expires: 0");

// Set default value if not passed
if (!isset($useVideo)) {
    $useVideo = false;
}

// Sliding Gallery Configuration
$galleryConfig = [
    'direction' => 'left-to-right', // 'left-to-right' or 'right-to-left'
    'speed' => 60, // seconds for one complete cycle
    'itemWidth' => 400, // width of each item in pixels
    'itemHeight' => 300, // height of each item in pixels
    'gap' => 30, // gap between items in pixels
    'responsiveScaling' => true, // Enable responsive scaling for large screens
];

// Gallery Items - Add your images and videos here
$galleryItems = [
    ['type' => 'video', 'src' => 'assets/video/bc_vid1.mp4', 'poster' => 'assets/video/bc_1.webp'],
    ['type' => 'image', 'src' => 'assets/video/bc_1.webp', 'alt' => 'BestCall Project'],
    ['type' => 'video', 'src' => 'assets/video/bc_vid2.mp4', 'poster' => 'assets/video/bc_2a.webp'],
    ['type' => 'image', 'src' => 'assets/video/bc_2a.webp', 'alt' => 'BestCall Project'],
    ['type' => 'image', 'src' => 'assets/video/bc_2.webp', 'alt' => 'BestCall Project'],
    ['type' => 'video', 'src' => 'assets/video/bc_vid3.mp4', 'poster' => 'assets/video/bc_3.webp'],
    ['type' => 'image', 'src' => 'assets/video/bc_3.webp', 'alt' => 'BestCall Project'],
    ['type' => 'image', 'src' => 'assets/video/bc_4.webp', 'alt' => 'BestCall Project'],
    ['type' => 'video', 'src' => 'assets/video/bc_vid4.mp4', 'poster' => 'assets/video/bc_5.webp'],
    ['type' => 'image', 'src' => 'assets/video/bc_5.webp', 'alt' => 'BestCall Project'],
];
?>

<!-- Hero Section with Sliding Gallery -->
<div class="hero-section" id="heroSection">
    <?php if ($useVideo): ?>
        <!-- Original Video Background (fallback) -->
        <div class="hero-background-video" style="display: none;">
            <video poster="<?= base_url('assets/video/battle_still.png') ?>" 
                   playsinline="playsinline" 
                   autoplay="autoplay" 
                   muted="muted" 
                   loop="loop">
                <source src="<?= base_url('assets/video/RRSH.mp4') ?>" type="video/mp4">
                <p class="warning">Your browser does not support HTML5 video.</p>
            </video>
        </div>
    <?php else: ?>
        <!-- Sliding Gallery Background -->
        <div class="hero-sliding-gallery">
            <!-- Background Pattern -->
            <div class="gallery-background"></div>
            
            <!-- Sliding Container -->
            <div class="gallery-slider" 
                 data-direction="<?= $galleryConfig['direction'] ?>"
                 data-speed="<?= $galleryConfig['speed'] ?>"
                 style="--item-width: <?= $galleryConfig['itemWidth'] ?>px; --item-height: <?= $galleryConfig['itemHeight'] ?>px; --gap: <?= $galleryConfig['gap'] ?>px;">
                
                <!-- Continuous Track with 3 Sets -->
                <div class="gallery-track">
                    <?php 
                    // Create 3 sets for truly seamless loop
                    for ($set = 0; $set < 3; $set++): 
                        foreach ($galleryItems as $index => $item): ?>
                            <div class="gallery-item">
                                <?php if ($item['type'] === 'video'): ?>
                                    <video class="gallery-video" 
                                           autoplay 
                                           muted 
                                           loop 
                                           playsinline
                                           poster="<?= base_url($item['poster']) ?>">
                                        <source src="<?= base_url($item['src']) ?>" type="video/mp4">
                                    </video>
                                <?php else: ?>
                                    <img class="gallery-image" 
                                         src="<?= base_url($item['src']) ?>" 
                                         alt="<?= $item['alt'] ?>"
                                         loading="lazy">
                                <?php endif; ?>
                            </div>
                        <?php endforeach; 
                    endfor; ?>
                </div>
            </div>
        </div>
        
        <!-- Static Fallback Background -->
        <div class="hero-background-static"></div>
    <?php endif; ?>
    
    <!-- Hero Overlay -->
    <div class="hero-overlay"></div>
    
    <!-- Hero Content -->
    <div class="hero-content">
        <!-- Minimal Search Container -->
        <div class="minimal-search-container">
            <!-- Tab Headers -->
            <div class="search-tabs">
                <button type="button" class="search-tab active" id="zip-tab" data-mode="zip">
                    ZIP Code
                </button>
                <button type="button" class="search-tab" id="address-tab" data-mode="address">
                    Address
                </button>
            </div>
            
            <!-- Search Form -->
            <form id="hero-search-form" class="hero-form">
                <!-- Hidden Fields -->
                <input type="hidden" id="form_latitude" name="latitude" value="">
                <input type="hidden" id="form_longitude" name="longitude" value="">
                <input type="hidden" id="selected_category" name="category" value="">
                <input type="hidden" id="form_keyword" name="keyword" value="">
                <input type="hidden" id="form_zipcode" name="zipcode" value="">
                <input type="hidden" id="search_mode" name="search_mode" value="zip">
                
                <!-- ZIP Code Mode -->
                <div class="search-input-container" id="zip-search-mode">
                    <div class="zip-input-wrapper">
                        <button type="button" class="location-btn" id="crosshairs-btn" title="Use my current location">
                            <i class="fas fa-crosshairs"></i>
                        </button>
                        
                        <div class="zip-digits">
                            <input type="text" class="zip-digit" maxlength="1" pattern="[0-9]" inputmode="numeric" id="zip-1">
                            <input type="text" class="zip-digit" maxlength="1" pattern="[0-9]" inputmode="numeric" id="zip-2">
                            <input type="text" class="zip-digit" maxlength="1" pattern="[0-9]" inputmode="numeric" id="zip-3">
                            <input type="text" class="zip-digit" maxlength="1" pattern="[0-9]" inputmode="numeric" id="zip-4">
                            <input type="text" class="zip-digit" maxlength="1" pattern="[0-9]" inputmode="numeric" id="zip-5">
                        </div>
                        
                        <button type="submit" class="search-submit-btn" id="zip-submit-btn">
                            <i class="fas fa-search"></i>
                        </button>
                    </div>
                </div>

                <!-- Address Mode -->
                <div class="search-input-container" id="address-search-mode" style="display: none;">
                    <div class="address-input-wrapper">
                        <button type="button" class="location-btn" id="address-crosshairs-btn" title="Use my current location">
                            <i class="fas fa-crosshairs"></i>
                        </button>
                        
                        <input type="text" 
                               id="address_search_input" 
                               class="address-input" 
                               placeholder="Enter your address..."
                               autocomplete="off">
                        
                        <button type="submit" class="search-submit-btn" id="address-submit-btn">
                            <i class="fas fa-search"></i>
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
/* CSS Variables */
:root {
    --primary-color: #667eea;
    --primary-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    --secondary-color: #ff6b6b;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --light-color: #f8f9fa;
    --dark-color: #212529;
    
    --border-radius: 20px;
    --border-radius-lg: 25px;
    --border-radius-sm: 15px;
    
    --shadow-sm: 0 2px 10px rgba(0,0,0,0.08);
    --shadow-md: 0 4px 20px rgba(0,0,0,0.12);
    --shadow-lg: 0 8px 30px rgba(0,0,0,0.15);
    --shadow-xl: 0 12px 40px rgba(0,0,0,0.18);
    
    --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
}

/* Hero Section */
.hero-section {
    position: relative;
    height: calc(50vh - 120px); /* Adjusted for actual sticky header height */
    min-height: 400px;
    max-height: 600px;
    display: flex;
    align-items: center;
    justify-content: center;
    overflow: hidden;
    z-index: 1;
    margin-top: 120px; /* Push down from sticky header */
}

/* Large Desktop Adjustments */
@media (min-width: 1920px) {
    .hero-section {
        height: calc(45vh - 120px); /* Slightly smaller on very large screens */
        max-height: 700px; /* Allow taller on large screens */
    }
}

@media (min-width: 2560px) {
    .hero-section {
        height: calc(40vh - 120px); /* Even more conservative on ultra-wide */
        max-height: 800px;
    }
}

/* Sliding Gallery Background */
.hero-sliding-gallery {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    z-index: 1;
    overflow: hidden;
}

.gallery-background {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: 
        radial-gradient(circle at 20% 80%, rgba(102, 126, 234, 0.1) 0%, transparent 50%),
        radial-gradient(circle at 80% 20%, rgba(118, 75, 162, 0.1) 0%, transparent 50%),
        linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
    z-index: 1;
}

.gallery-slider {
    position: absolute;
    top: 50%;
    left: 0;
    height: calc(var(--item-height) + 40px); /* Increased padding for scaled items */
    transform: translateY(-50%);
    display: flex;
    z-index: 2;
    width: 100vw;
    overflow: hidden;
}

.gallery-track {
    display: flex;
    gap: calc(var(--gap) + 10px); /* Increased gap instead of using margin */
    animation: slideGallery var(--animation-duration, 60s) linear infinite;
    padding: 0 10px;
    white-space: nowrap;
}

.gallery-item {
    width: var(--item-width);
    height: var(--item-height);
    border-radius: 15px;
    overflow: hidden;
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
    border: 4px solid rgba(255, 255, 255, 0.95);
    background: white;
    flex-shrink: 0;
    position: relative;
}

.gallery-item::before {
    content: '';
    position: absolute;
    top: -2px;
    left: -2px;
    right: -2px;
    bottom: -2px;
    background: linear-gradient(45deg, #667eea, #764ba2, #ff6b6b, #667eea);
    border-radius: 18px;
    z-index: -1;
    opacity: 0.7;
}

.gallery-image,
.gallery-video {
    width: 100%;
    height: 100%;
    object-fit: cover;
    border-radius: 12px;
}

/* Animation Keyframes - Simple continuous scroll */
@keyframes slideGallery {
    0% {
        transform: translateX(0);
    }
    100% {
        transform: translateX(calc(-33.333%)); /* Move exactly 1/3 since we have 3 sets */
    }
}

/* Direction Control */
.gallery-slider[data-direction="right-to-left"] .gallery-track {
    animation-direction: normal;
}

.gallery-slider[data-direction="left-to-right"] .gallery-track {
    animation-direction: reverse;
}

/* Responsive scaling for larger screens */
@media (min-width: 1440px) {
    .gallery-item {
        width: calc(var(--item-width) * 1.1);
        height: calc(var(--item-height) * 1.1);
    }
}

@media (min-width: 1920px) {
    .gallery-item {
        width: calc(var(--item-width) * 1.25);
        height: calc(var(--item-height) * 1.25);
    }
}

@media (min-width: 2560px) {
    .gallery-item {
        width: calc(var(--item-width) * 1.5);
        height: calc(var(--item-height) * 1.5);
        border-radius: 20px;
    }
}

@media (min-width: 3200px) {
    .gallery-item {
        width: calc(var(--item-width) * 1.75);
        height: calc(var(--item-height) * 1.75);
        border-radius: 25px;
    }
}

/* Original Background Elements */
.hero-background-video {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    z-index: 1;
}

.hero-background-video video {
    position: absolute;
    top: 50%;
    left: 50%;
    min-width: 100%;
    min-height: 100%;
    width: auto;
    height: auto;
    transform: translate(-50%, -50%);
    z-index: 0;
}

.hero-background-static {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: url('https://bestcallhvac.com/front.webp') center/cover no-repeat;
    background-attachment: scroll;
    background-position: center center;
    z-index: 0;
    display: none;
}

/* Show static background as fallback on very small screens */
@media (max-width: 576px) {
    .hero-sliding-gallery {
        display: none;
    }
    
    .hero-background-static {
        display: block !important;
        z-index: 1;
    }
}

/* Subtle Overlay */
.hero-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: linear-gradient(135deg, rgba(0, 0, 0, 0.3) 0%, rgba(0, 0, 0, 0.2) 100%);
    z-index: 3;
}

/* Hero Content */
.hero-content {
    position: relative;
    z-index: 4;
    text-align: center;
    width: 100%;
    max-width: 600px;
    padding: 2rem;
    margin: 0 auto;
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
}

/* Minimal Search Container */
.minimal-search-container {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0;
    position: relative;
}

/* Search Tabs */
.search-tabs {
    display: flex;
    background: transparent;
    padding: 0;
    margin-bottom: -1px;
    z-index: 3;
}

.search-tab {
    background: rgba(255, 255, 255, 0.7);
    border: none;
    color: rgba(102, 126, 234, 0.8);
    padding: 0.75rem 2rem;
    border-radius: 15px 15px 0 0;
    cursor: pointer;
    transition: var(--transition);
    font-weight: 500;
    font-size: 0.95rem;
    white-space: nowrap;
    border: 1px solid rgba(255, 255, 255, 0.3);
    border-bottom: none;
    backdrop-filter: blur(10px);
}

.search-tab:hover {
    background: rgba(255, 255, 255, 0.85);
    color: var(--primary-color);
    transform: translateY(-2px);
}

.search-tab.active {
    background: rgba(255, 255, 255, 0.95);
    color: var(--primary-color);
    font-weight: 600;
    box-shadow: 0 -2px 10px rgba(0, 0, 0, 0.1);
    border-color: rgba(255, 255, 255, 0.5);
    transform: translateY(-2px);
}

/* Search Input Container */
.search-input-container {
    width: 100%;
    max-width: 500px;
}

/* ZIP Code Input */
.zip-input-wrapper {
    display: flex;
    align-items: center;
    background: rgba(255, 255, 255, 0.95);
    border-radius: 25px;
    padding: 0.75rem;
    backdrop-filter: blur(15px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
    border: 1px solid rgba(255, 255, 255, 0.3);
    gap: 1rem;
}

.zip-digits {
    display: flex;
    gap: 0.5rem;
    flex: 1;
    justify-content: center;
}

.zip-digit {
    width: 3rem;
    height: 3rem;
    border: 2px solid #e9ecef;
    border-radius: 12px;
    text-align: center;
    font-size: 1.25rem;
    font-weight: 700;
    color: var(--dark-color);
    background: white;
    transition: all 0.2s ease;
    outline: none;
}

.zip-digit:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    transform: scale(1.05);
}

.zip-digit.filled {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.zip-digit.error {
    border-color: var(--danger-color);
    background: rgba(220, 53, 69, 0.1);
    animation: shake 0.3s ease-in-out;
}

/* Address Input */
.address-input-wrapper {
    display: flex;
    align-items: center;
    background: rgba(255, 255, 255, 0.95);
    border-radius: 25px;
    padding: 0.75rem;
    backdrop-filter: blur(15px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
    border: 1px solid rgba(255, 255, 255, 0.3);
    gap: 1rem;
}

.address-input {
    flex: 1;
    border: none;
    outline: none;
    font-size: 1.1rem;
    padding: 0.75rem 1rem;
    background: transparent;
    color: var(--dark-color);
    border-radius: 15px;
}

.address-input::placeholder {
    color: #666;
    opacity: 0.8;
}

/* Action Buttons */
.location-btn,
.search-submit-btn {
    width: 50px;
    height: 50px;
    border: none;
    border-radius: 50%;
    cursor: pointer;
    transition: var(--transition);
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.1rem;
    flex-shrink: 0;
}

.location-btn {
    background: var(--primary-color);
    color: white;
}

.location-btn:hover {
    background: var(--secondary-color);
    transform: scale(1.05);
}

.search-submit-btn {
    background: var(--primary-gradient);
    color: white;
}

.search-submit-btn:hover {
    transform: scale(1.05);
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

/* Animations */
@keyframes shake {
    0%, 100% { transform: translateX(0); }
    25% { transform: translateX(-2px); }
    75% { transform: translateX(2px); }
}

/* Mobile Responsiveness */
@media (max-width: 768px) {
    .hero-section {
        margin-top: 80px; /* Smaller margin for mobile since logo is hidden */
        height: calc(40vh - 80px); /* Adjusted for mobile header */
    }
    
    .hero-content {
        padding: 1rem;
        max-width: 95%;
    }
    
    .search-tab {
        padding: 0.625rem 1.5rem;
        font-size: 0.9rem;
    }
    
    .zip-digit {
        width: 2.5rem;
        height: 2.5rem;
        font-size: 1.1rem;
    }
    
    .zip-digits {
        gap: 0.4rem;
    }
    
    .location-btn,
    .search-submit-btn {
        width: 45px;
        height: 45px;
        font-size: 1rem;
    }
    
    .address-input {
        font-size: 1rem;
        padding: 0.625rem 0.875rem;
    }
    
    .gallery-item {
        width: 280px !important;
        height: 200px !important;
    }
}

@media (max-width: 576px) {
    .search-tab {
        padding: 0.5rem 1.25rem;
        font-size: 0.85rem;
    }
    
    .zip-digit {
        width: 2.2rem;
        height: 2.2rem;
        font-size: 1rem;
    }
    
    .zip-digits {
        gap: 0.3rem;
    }
    
    .location-btn,
    .search-submit-btn {
        width: 40px;
        height: 40px;
        font-size: 0.95rem;
    }
    
    .zip-input-wrapper,
    .address-input-wrapper {
        padding: 0.625rem;
        gap: 0.75rem;
    }
    
    .gallery-item {
        width: 240px !important;
        height: 180px !important;
        border-radius: 12px !important;
    }
}
</style>

<script>
// Enhanced Minimal Search System with Gallery Controls
document.addEventListener('DOMContentLoaded', function() {
    setupMinimalSearchToggle();
    setupZipCodeHandlers();
    setupAddressHandlers();
    setupSlidingGallery();
});

function setupSlidingGallery() {
    const gallerySlider = document.querySelector('.gallery-slider');
    if (!gallerySlider) return;
    
    const direction = gallerySlider.dataset.direction;
    const speed = parseInt(gallerySlider.dataset.speed) || 60;
    
    // Set CSS custom property for animation duration
    gallerySlider.style.setProperty('--animation-duration', speed + 's');
    
    // Pause animation on hover (optional)
    gallerySlider.addEventListener('mouseenter', function() {
        this.style.animationPlayState = 'paused';
    });
    
    gallerySlider.addEventListener('mouseleave', function() {
        this.style.animationPlayState = 'running';
    });
}

function setupMinimalSearchToggle() {
    const zipTab = document.getElementById('zip-tab');
    const addressTab = document.getElementById('address-tab');
    const zipSearchMode = document.getElementById('zip-search-mode');
    const addressSearchMode = document.getElementById('address-search-mode');
    const searchModeField = document.getElementById('search_mode');
    
    function switchToZipMode() {
        zipTab.classList.add('active');
        addressTab.classList.remove('active');
        zipSearchMode.style.display = 'block';
        addressSearchMode.style.display = 'none';
        searchModeField.value = 'zip';
        
        const addressInput = document.getElementById('address_search_input');
        if (addressInput) addressInput.value = '';
        
        setTimeout(() => {
            const firstZip = document.getElementById('zip-1');
            if (firstZip) firstZip.focus();
        }, 100);
    }
    
    function switchToAddressMode() {
        addressTab.classList.add('active');
        zipTab.classList.remove('active');
        addressSearchMode.style.display = 'block';
        zipSearchMode.style.display = 'none';
        searchModeField.value = 'address';
        
        clearZipCode();
        
        setTimeout(() => {
            const addressInput = document.getElementById('address_search_input');
            if (addressInput) addressInput.focus();
        }, 100);
    }
    
    zipTab.addEventListener('click', switchToZipMode);
    addressTab.addEventListener('click', switchToAddressMode);
    
    switchToZipMode();
}

function setupZipCodeHandlers() {
    const zipDigits = document.querySelectorAll('.zip-digit');
    
    zipDigits.forEach((digit, index) => {
        digit.addEventListener('input', function(e) {
            const value = e.target.value;
            
            if (!/^\d$/.test(value) && value !== '') {
                e.target.value = '';
                e.target.classList.add('error');
                setTimeout(() => e.target.classList.remove('error'), 300);
                return;
            }
            
            if (value) {
                e.target.classList.add('filled');
                if (index < zipDigits.length - 1) {
                    zipDigits[index + 1].focus();
                }
            } else {
                e.target.classList.remove('filled');
            }
            
            updateZipState();
        });
        
        digit.addEventListener('keydown', function(e) {
            if (e.key === 'Backspace' && !e.target.value && index > 0) {
                zipDigits[index - 1].focus();
                zipDigits[index - 1].value = '';
                zipDigits[index - 1].classList.remove('filled');
                updateZipState();
            }
        });
        
        digit.addEventListener('paste', function(e) {
            e.preventDefault();
            const pastedData = e.clipboardData.getData('text').replace(/\D/g, '');
            
            if (pastedData.length === 5) {
                zipDigits.forEach((input, i) => {
                    input.value = pastedData[i] || '';
                    if (pastedData[i]) {
                        input.classList.add('filled');
                    } else {
                        input.classList.remove('filled');
                    }
                });
                updateZipState();
            }
        });
    });
    
    function updateZipState() {
        const zipValue = Array.from(zipDigits).map(d => d.value).join('');
        const zipField = document.getElementById('form_zipcode');
        if (zipField) zipField.value = zipValue;
    }
}

function setupAddressHandlers() {
    const crosshairsBtn = document.getElementById('crosshairs-btn');
    const addressCrosshairsBtn = document.getElementById('address-crosshairs-btn');
    
    if (crosshairsBtn) {
        crosshairsBtn.addEventListener('click', function(e) {
            e.preventDefault();
            if (typeof getLocation === 'function') {
                getLocation();
            } else {
                alert('Location service not available');
            }
        });
    }
    
    if (addressCrosshairsBtn) {
        addressCrosshairsBtn.addEventListener('click', function(e) {
            e.preventDefault();
            if (typeof getLocation === 'function') {
                getLocation();
            } else {
                alert('Location service not available');
            }
        });
    }
}

function setupFormSubmission() {
    const form = document.getElementById('hero-search-form');
    if (form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const searchMode = document.getElementById('search_mode').value;
            
            if (searchMode === 'zip') {
                handleZipSearch();
            } else if (searchMode === 'address') {
                handleAddressSearch();
            }
        });
    }
}

function clearZipCode() {
    const zipDigits = document.querySelectorAll('.zip-digit');
    zipDigits.forEach(digit => {
        digit.value = '';
        digit.classList.remove('filled');
    });
    const zipField = document.getElementById('form_zipcode');
    if (zipField) zipField.value = '';
}

function handleZipSearch() {
    const zipCode = document.getElementById('form_zipcode')?.value || '';
    
    if (zipCode.length !== 5) {
        alert('Please enter a complete 5-digit ZIP code.');
        document.getElementById('zip-1')?.focus();
        return;
    }
    
    console.log('ZIP search:', zipCode);
    
    if (typeof geocodeZipAndSearch === 'function') {
        geocodeZipAndSearch(zipCode);
    } else {
        alert('Search system not loaded. Please refresh the page.');
    }
}

function handleAddressSearch() {
    const address = document.getElementById('address_search_input')?.value.trim();
    
    if (!address) {
        alert('Please enter an address to search.');
        document.getElementById('address_search_input')?.focus();
        return;
    }
    
    console.log('Address search:', address);
    
    if (typeof geocodeAddressAndSearch === 'function') {
        geocodeAddressAndSearch(address);
    } else {
        alert('Search system not loaded. Please refresh the page.');
    }
}
</script>