<?php
/**
 * Site Structure Diagnostic Tool for Claude AI Analysis
 * Place in root directory and access via browser
 * Provides detailed application architecture overview for AI-assisted development
 */

// File extensions to exclude
$excludeExtensions = [
    'log', 'tmp', 'cache', 'bak', 'old', 'backup', 
    'zip', 'tar', 'gz', 'rar', 'jpg', 'jpeg', 'png', 
    'gif', 'svg', 'ico', 'pdf', 'doc', 'docx', 'xls', 
    'xlsx', 'ppt', 'pptx', 'mp3', 'mp4', 'avi', 'mov'
];

// Directories to exclude
$excludeDirs = [
    '.git', '.svn', '.idea', 'node_modules', 'vendor/bin',
    'writable/logs', 'writable/cache', 'writable/session',
    'writable/uploads', 'public/uploads', 'storage/logs'
];

// Key file types for Claude analysis
$keyFiles = [
    'php' => 'Controllers, Models, Views, Config',
    'sql' => 'Database Schema, Migrations',
    'js' => 'Frontend Logic, API Calls', 
    'css' => 'Styling, Layout Components',
    'json' => 'Configuration, Package Management',
    'md' => 'Documentation, API Specs',
    'htaccess' => 'Server Config, URL Rewriting',
    'xml' => 'Configuration, Data Exchange',
    'txt' => 'Documentation, Notes'
];

/**
 * Scan directory recursively
 */
function scanDirectory($dir, $basePath = '', $level = 0) {
    global $excludeDirs, $excludeExtensions;
    
    if ($level > 10) return []; // Prevent infinite recursion
    
    $items = [];
    $path = $basePath ? $basePath . '/' . $dir : $dir;
    
    // Skip excluded directories
    foreach ($excludeDirs as $excludeDir) {
        if (strpos($path, $excludeDir) !== false) {
            return [];
        }
    }
    
    if (!is_dir($path) || !is_readable($path)) {
        return [];
    }
    
    try {
        $files = scandir($path);
        if ($files === false) return [];
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') continue;
            
            $fullPath = $path . '/' . $file;
            $relativePath = $basePath ? $basePath . '/' . $dir . '/' . $file : $dir . '/' . $file;
            
            if (is_dir($fullPath)) {
                // Skip hidden and excluded directories
                if (strpos($file, '.') === 0) continue;
                
                $shouldExclude = false;
                foreach ($excludeDirs as $excludeDir) {
                    if (strpos($relativePath, $excludeDir) !== false) {
                        $shouldExclude = true;
                        break;
                    }
                }
                
                if (!$shouldExclude) {
                    $items[] = [
                        'name' => $file,
                        'type' => 'directory',
                        'path' => $relativePath,
                        'level' => $level,
                        'children' => scanDirectory($file, $path, $level + 1)
                    ];
                }
            } else {
                // Check file extension
                $ext = strtolower(pathinfo($file, PATHINFO_EXTENSION));
                if (!in_array($ext, $excludeExtensions)) {
                    $items[] = [
                        'name' => $file,
                        'type' => 'file',
                        'extension' => $ext,
                        'path' => $relativePath,
                        'level' => $level,
                        'size' => is_readable($fullPath) ? filesize($fullPath) : 0
                    ];
                }
            }
        }
    } catch (Exception $e) {
        // Skip directories that can't be read
    }
    
    return $items;
}

/**
 * Format file size
 */
function formatSize($bytes) {
    if ($bytes >= 1073741824) {
        return number_format($bytes / 1073741824, 2) . ' GB';
    } elseif ($bytes >= 1048576) {
        return number_format($bytes / 1048576, 2) . ' MB';
    } elseif ($bytes >= 1024) {
        return number_format($bytes / 1024, 2) . ' KB';
    } else {
        return $bytes . ' B';
    }
}

/**
 * Get application architecture insights
 */
function getArchitectureInsights() {
    $insights = [];
    
    // Database integration
    if (file_exists('app/Config/Database.php')) {
        $insights['database'] = 'CodeIgniter 4 Database Configuration';
    }
    
    // Authentication system
    $authTypes = [];
    if (is_dir('app/Controllers/Business')) {
        $authTypes[] = 'Business Portal';
    }
    if (is_dir('app/Controllers/Admin')) {
        $authTypes[] = 'Admin Panel';
    }
    if (file_exists('app/Controllers/Auth.php')) {
        $authTypes[] = 'User Authentication';
    }
    if ($authTypes) {
        $insights['authentication'] = implode(', ', $authTypes);
    }
    
    // API endpoints
    if (is_dir('app/Controllers/API') || file_exists('app/Controllers/Api.php')) {
        $insights['api'] = 'REST API Implementation';
    }
    
    // Frontend frameworks
    $frontendFiles = [];
    if (file_exists('public/js/app.js') || file_exists('public/assets/js/app.js')) {
        $frontendFiles[] = 'Custom JavaScript';
    }
    if (file_exists('package.json')) {
        $frontendFiles[] = 'Node.js Dependencies';
    }
    if (glob('public/css/*.css') || glob('public/assets/css/*.css')) {
        $frontendFiles[] = 'Custom Stylesheets';
    }
    if ($frontendFiles) {
        $insights['frontend'] = implode(', ', $frontendFiles);
    }
    
    // Special features
    if (is_dir('app/Controllers/Business/Onboarding.php') || file_exists('app/Controllers/Business/Onboarding.php')) {
        $insights['onboarding'] = 'Business Onboarding System';
    }
    
    return $insights;
}

/**
 * Get file type color
 */
function getFileColor($extension) {
    $colors = [
        'php' => '#8892BF',
        'js' => '#F7DF1E', 
        'css' => '#1572B6',
        'html' => '#E34F26',
        'sql' => '#336791',
        'json' => '#000000',
        'xml' => '#FF6600',
        'md' => '#083FA1',
        'txt' => '#888888',
        'htaccess' => '#D14836'
    ];
    
    return $colors[$extension] ?? '#666666';
}

/**
 * Render directory tree
 */
function renderTree($items, $level = 0) {
    foreach ($items as $item) {
        $indent = str_repeat('  ', $level);
        $icon = $item['type'] === 'directory' ? '📁' : '📄';
        $color = $item['type'] === 'file' ? getFileColor($item['extension']) : '#333';
        $size = $item['type'] === 'file' ? ' (' . formatSize($item['size']) . ')' : '';
        
        echo "<div style='margin-left: " . ($level * 20) . "px; color: $color; font-family: monospace;'>";
        echo "<span style='color: #666;'>$icon</span> " . htmlspecialchars($item['name']) . $size;
        echo "</div>\n";
        
        if ($item['type'] === 'directory' && !empty($item['children'])) {
            renderTree($item['children'], $level + 1);
        }
    }
}

// Get site structure
$siteStructure = scanDirectory('.');

// Count files by type
$fileCounts = [];
$totalSize = 0;
function countFiles($items) {
    global $fileCounts, $totalSize;
    foreach ($items as $item) {
        if ($item['type'] === 'file') {
            $ext = $item['extension'] ?: 'no-extension';
            $fileCounts[$ext] = ($fileCounts[$ext] ?? 0) + 1;
            $totalSize += $item['size'];
        }
        if ($item['type'] === 'directory' && !empty($item['children'])) {
            countFiles($item['children']);
        }
    }
}
countFiles($siteStructure);

// Sort file counts
arsort($fileCounts);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Site Structure Diagnostic</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Arial, sans-serif;
            line-height: 1.6;
            margin: 0;
            padding: 20px;
            background: #f5f5f5;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: white;
            padding: 30px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #333;
            border-bottom: 3px solid #10b981;
            padding-bottom: 10px;
        }
        .summary {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin: 20px 0;
        }
        .card {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 6px;
            border-left: 4px solid #10b981;
        }
        .card h3 {
            margin: 0 0 10px 0;
            color: #333;
        }
        .file-tree {
            background: #fafafa;
            padding: 20px;
            border-radius: 6px;
            border: 1px solid #ddd;
            max-height: 600px;
            overflow-y: auto;
            font-size: 14px;
        }
        .file-counts {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: 10px;
            margin: 20px 0;
        }
        .count-item {
            background: #f8f9fa;
            padding: 10px;
            text-align: center;
            border-radius: 4px;
            border: 1px solid #ddd;
        }
        .count-number {
            font-size: 1.5em;
            font-weight: bold;
            color: #10b981;
        }
        .warning {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            color: #856404;
            padding: 15px;
            border-radius: 6px;
            margin: 20px 0;
        }
        .key-files {
            margin: 20px 0;
        }
        .key-files ul {
            columns: 2;
            column-gap: 40px;
        }
        @media (max-width: 768px) {
            .key-files ul {
                columns: 1;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔍 Site Structure Diagnostic</h1>
        
        <div class="warning">
            <strong>⚠️ Development Tool:</strong> This diagnostic tool should only be used in development/staging environments. 
            Remove from production servers.
        </div>

        <div class="summary">
            <div class="card">
                <h3>📊 Overview</h3>
                <p><strong>Total Files:</strong> <?= array_sum($fileCounts) ?></p>
                <p><strong>Total Size:</strong> <?= formatSize($totalSize) ?></p>
                <p><strong>File Types:</strong> <?= count($fileCounts) ?></p>
                <p><strong>Generated:</strong> <?= date('Y-m-d H:i:s') ?></p>
            </div>
            
            <div class="card">
                <h3>🏗️ Framework Detection</h3>
                <p><strong>Type:</strong> <?= file_exists('app/Config/Routes.php') ? 'CodeIgniter 4' : 'Unknown' ?></p>
                <p><strong>App Directory:</strong> <?= is_dir('app') ? '✅ Found' : '❌ Missing' ?></p>
                <p><strong>Public Directory:</strong> <?= is_dir('public') ? '✅ Found' : '❌ Missing' ?></p>
                <p><strong>Writable Directory:</strong> <?= is_dir('writable') ? '✅ Found' : '❌ Missing' ?></p>
            </div>
            
            <div class="card">
                <h3>🔧 Key Components</h3>
                <p><strong>Controllers:</strong> <?= is_dir('app/Controllers') ? '✅ Found' : '❌ Missing' ?></p>
                <p><strong>Models:</strong> <?= is_dir('app/Models') ? '✅ Found' : '❌ Missing' ?></p>
                <p><strong>Views:</strong> <?= is_dir('app/Views') ? '✅ Found' : '❌ Missing' ?></p>
                <p><strong>Config:</strong> <?= is_dir('app/Config') ? '✅ Found' : '❌ Missing' ?></p>
            </div>
        </div>

        <h2>📁 File Type Distribution</h2>
        <div class="file-counts">
            <?php foreach ($fileCounts as $ext => $count): ?>
                <div class="count-item">
                    <div class="count-number"><?= $count ?></div>
                    <div>.<?= $ext ?></div>
                </div>
            <?php endforeach; ?>
        </div>

        <div class="key-files">
            <h2>🔑 Key File Types</h2>
            <ul>
                <?php foreach ($keyFiles as $ext => $description): ?>
                    <li><strong>.<?= $ext ?>:</strong> <?= $description ?> 
                        <?= isset($fileCounts[$ext]) ? "({$fileCounts[$ext]} files)" : "(0 files)" ?>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>

        <h2>🌲 Directory Structure</h2>
        <div class="file-tree">
            <?php renderTree($siteStructure); ?>
        </div>
        
        <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; color: #666; font-size: 12px;">
            Diagnostic completed in <?= round((microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']) * 1000) ?>ms
        </div>
    </div>
</body>
</html>