<?php
/**
 * Focused Admin Structure Analyzer
 * 
 * This script generates only the essential admin and database information needed
 * for business portal development planning
 */

// Prevent direct access in production
if ($_SERVER['HTTP_HOST'] !== 'localhost' && !isset($_GET['allow'])) {
    die('Access denied. Add ?allow=true for non-localhost access.');
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Structure Analysis</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; }
        .section { margin-bottom: 30px; border: 1px solid #ddd; padding: 15px; border-radius: 5px; }
        .section h2 { margin-top: 0; color: #333; border-bottom: 2px solid #007bff; padding-bottom: 10px; }
        .code-block { background: #f8f9fa; padding: 15px; border-radius: 4px; font-family: monospace; font-size: 12px; }
        .table-structure { border-collapse: collapse; width: 100%; margin: 10px 0; }
        .table-structure th, .table-structure td { border: 1px solid #ddd; padding: 8px; text-align: left; font-size: 12px; }
        .table-structure th { background: #f8f9fa; }
        .method-list { background: #e3f2fd; padding: 10px; border-radius: 4px; margin: 5px 0; }
        .warning { background: #fff3cd; border: 1px solid #ffeaa7; padding: 10px; border-radius: 4px; margin: 10px 0; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🎯 Focused Admin Structure Analysis</h1>
        <div class="warning">
            <strong>⚠️ Security Notice:</strong> Remove this file after analysis!
        </div>

<?php

// Get the application root directory
$appRoot = dirname(__DIR__);

// Database Analysis - Essential Tables Only
echo "<div class='section'>";
echo "<h2>🗄️ Database Tables (Essential for Business Portal)</h2>";
analyzeFocusedDatabase();
echo "</div>";

// Admin Controllers Analysis
echo "<div class='section'>";
echo "<h2>🎮 Admin Controllers Structure</h2>";
analyzeAdminControllers($appRoot . '/app/Controllers');
echo "</div>";

// Authentication Analysis
echo "<div class='section'>";
echo "<h2>🔐 Authentication & Session Structure</h2>";
analyzeAuthentication($appRoot);
echo "</div>";

// Routes Analysis - Admin Only
echo "<div class='section'>";
echo "<h2>🛣️ Admin Routes</h2>";
analyzeAdminRoutes($appRoot . '/app/Config/Routes.php');
echo "</div>";

// Models Analysis
echo "<div class='section'>";
echo "<h2>📊 Models for Business Portal Integration</h2>";
analyzeRelevantModels($appRoot . '/app/Models');
echo "</div>";

/**
 * Analyze only essential database tables
 */
function analyzeFocusedDatabase() {
    try {
        // Try multiple ways to get database config
        $configPath = dirname(__DIR__) . '/app/Config/Database.php';
        
        if (!file_exists($configPath)) {
            echo "<p>❌ Database config file not found at: {$configPath}</p>";
            return;
        }
        
        // Read the config file content to show what's there
        $configContent = file_get_contents($configPath);
        echo "<h4>📄 Database Config File Found</h4>";
        
        // Try to extract database info without including the file
        if (preg_match('/[\'"]database[\'"][^=]*=>[^\'"]*([\'"]\w+[\'"])/', $configContent, $matches)) {
            $dbName = trim($matches[1], '"\'');
            echo "<p><strong>Database Name (from config):</strong> {$dbName}</p>";
        }
        
        if (preg_match('/[\'"]hostname[\'"][^=]*=>[^\'"]*([\'"]\w+[\'"])/', $configContent, $matches)) {
            $hostname = trim($matches[1], '"\'');
            echo "<p><strong>Hostname:</strong> {$hostname}</p>";
        }
        
        // Try direct PDO connection with environment variables or common defaults
        $dbConfigs = [
            ['localhost', 'root', '', 'business_directory'],
            ['localhost', 'root', 'root', 'business_directory'],
            ['127.0.0.1', 'root', '', 'business_directory'],
            ['localhost', getenv('DB_USERNAME') ?: 'root', getenv('DB_PASSWORD') ?: '', getenv('DB_DATABASE') ?: 'business_directory']
        ];
        
        $connected = false;
        $pdo = null;
        
        foreach ($dbConfigs as $config) {
            try {
                $dsn = "mysql:host={$config[0]};dbname={$config[3]}";
                $pdo = new PDO($dsn, $config[1], $config[2]);
                echo "<p>✅ <strong>Connected to database:</strong> {$config[3]} on {$config[0]}</p>";
                $connected = true;
                break;
            } catch (Exception $e) {
                // Try next config
                continue;
            }
        }
        
        if (!$connected) {
            echo "<p>❌ Could not connect to database with common configurations</p>";
            echo "<p>💡 <strong>Manual Check Needed:</strong> Please verify your database credentials</p>";
            return;
        }
        
        // Get all tables
        $stmt = $pdo->query("SHOW TABLES");
        $allTables = $stmt->fetchAll(PDO::FETCH_COLUMN);
        
        echo "<p><strong>📊 Total Tables Found:</strong> " . count($allTables) . "</p>";
        
        // Focus on business-relevant tables
        $importantTables = [
            'store', 'category', 'store_category', 'store_schedules', 
            'store_hour_types', 'store_current_status', 'business_verifications',
            'new_business_discoveries'
        ];
        
        $foundTables = array_intersect($importantTables, $allTables);
        echo "<p><strong>🎯 Essential Tables Found:</strong> " . count($foundTables) . "/" . count($importantTables) . "</p>";
        
        // Check for user/admin tables
        $userTables = array_filter($allTables, function($table) {
            return strpos(strtolower($table), 'user') !== false || 
                   strpos(strtolower($table), 'admin') !== false ||
                   strpos(strtolower($table), 'auth') !== false;
        });
        
        if (!empty($userTables)) {
            echo "<p><strong>🔐 User/Admin Tables:</strong> " . implode(', ', $userTables) . "</p>";
        } else {
            echo "<p>❌ No user/admin tables found</p>";
        }
        
        // Quick analysis of key tables
        foreach (['store'] as $table) {
            if (in_array($table, $allTables)) {
                quickTableAnalysis($pdo, $table);
            }
        }
        
        // Show all tables for reference
        echo "<h4>📋 All Tables in Database:</h4>";
        echo "<div class='code-block'>";
        foreach ($allTables as $index => $table) {
            echo $table;
            if ($index < count($allTables) - 1) echo ", ";
            if (($index + 1) % 8 == 0) echo "<br>"; // Line break every 8 tables
        }
        echo "</div>";
        
    } catch (Exception $e) {
        echo "<p>❌ Database analysis failed: " . $e->getMessage() . "</p>";
        echo "<p>💡 <strong>Next steps:</strong></p>";
        echo "<ul>";
        echo "<li>Check your database credentials in .env or config</li>";
        echo "<li>Ensure database server is running</li>";
        echo "<li>Verify database name exists</li>";
        echo "</ul>";
    }
}

/**
 * Quick table analysis without full structure
 */
function quickTableAnalysis($pdo, $table) {
    try {
        echo "<h4>📋 {$table} (Quick Analysis)</h4>";
        
        // Get column count
        $stmt = $pdo->query("DESCRIBE {$table}");
        $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
        echo "<p><strong>Columns:</strong> " . count($columns) . "</p>";
        
        // Get row count
        $stmt = $pdo->query("SELECT COUNT(*) as count FROM {$table}");
        $count = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        echo "<p><strong>Records:</strong> {$count}</p>";
        
        // Show key columns
        $keyColumns = array_filter($columns, function($col) {
            return !empty($col['Key']);
        });
        
        if (!empty($keyColumns)) {
            echo "<p><strong>Key Columns:</strong> ";
            foreach ($keyColumns as $col) {
                echo "{$col['Field']} ({$col['Key']}) ";
            }
            echo "</p>";
        }
        
    } catch (Exception $e) {
        echo "<p>❌ Could not analyze {$table}: " . $e->getMessage() . "</p>";
    }
}

/**
 * Analyze specific table structure
 */
function analyzeTableStructure($pdo, $table) {
    echo "<h4>📋 {$table}</h4>";
    
    $stmt = $pdo->query("DESCRIBE {$table}");
    $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "<table class='table-structure'>";
    echo "<tr><th>Field</th><th>Type</th><th>Key</th><th>Default</th></tr>";
    foreach ($columns as $column) {
        echo "<tr>";
        echo "<td><strong>{$column['Field']}</strong></td>";
        echo "<td>{$column['Type']}</td>";
        echo "<td>{$column['Key']}</td>";
        echo "<td>" . ($column['Default'] ?? 'NULL') . "</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    // Get row count
    $stmt = $pdo->query("SELECT COUNT(*) as count FROM {$table}");
    $count = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
    echo "<p><em>📊 Records: {$count}</em></p>";
}

/**
 * Analyze admin controllers
 */
function analyzeAdminControllers($controllersDir) {
    // Look for admin controllers
    $adminDir = $controllersDir . '/Admin';
    if (is_dir($adminDir)) {
        echo "<h4>📁 Admin Controllers Directory Found</h4>";
        $files = glob($adminDir . '/*.php');
        
        foreach ($files as $file) {
            $fileName = basename($file, '.php');
            echo "<h5>🎮 {$fileName}.php</h5>";
            
            $content = file_get_contents($file);
            
            // Extract methods
            preg_match_all('/public\s+function\s+(\w+)\s*\(([^)]*)\)/', $content, $matches);
            if (!empty($matches[1])) {
                echo "<div class='method-list'>";
                echo "<strong>Methods:</strong><br>";
                for ($i = 0; $i < count($matches[1]); $i++) {
                    $method = $matches[1][$i];
                    $params = trim($matches[2][$i]);
                    echo "• <code>{$method}(" . ($params ?: '') . ")</code><br>";
                }
                echo "</div>";
            }
        }
    } else {
        echo "<p>❌ No Admin controllers directory found</p>";
    }
    
    // Look for base controller or auth-related controllers
    $baseFiles = [
        $controllersDir . '/BaseController.php',
        $controllersDir . '/Auth.php',
        $controllersDir . '/Login.php',
        $controllersDir . '/Admin.php'
    ];
    
    foreach ($baseFiles as $file) {
        if (file_exists($file)) {
            $fileName = basename($file, '.php');
            echo "<h5>🔐 {$fileName}.php (Found)</h5>";
            
            $content = file_get_contents($file);
            preg_match_all('/public\s+function\s+(\w+)\s*\(([^)]*)\)/', $content, $matches);
            if (!empty($matches[1])) {
                echo "<div class='method-list'>";
                echo "<strong>Methods:</strong> " . implode(', ', $matches[1]);
                echo "</div>";
            }
        }
    }
}

/**
 * Analyze authentication structure
 */
function analyzeAuthentication($appRoot) {
    // Check for auth-related config files
    $authFiles = [
        $appRoot . '/app/Config/Auth.php',
        $appRoot . '/app/Config/Filters.php',
        $appRoot . '/app/Config/Validation.php'
    ];
    
    foreach ($authFiles as $file) {
        if (file_exists($file)) {
            $fileName = basename($file);
            echo "<p>✅ <strong>{$fileName}</strong> - Found</p>";
        } else {
            echo "<p>❌ <strong>{$fileName}</strong> - Not found</p>";
        }
    }
    
    // Check for session usage in controllers
    $controllersToCheck = glob($appRoot . '/app/Controllers/Admin/*.php');
    $sessionUsage = [];
    
    foreach ($controllersToCheck as $file) {
        $content = file_get_contents($file);
        if (strpos($content, 'session()') !== false || strpos($content, '$_SESSION') !== false) {
            $sessionUsage[] = basename($file, '.php');
        }
    }
    
    if (!empty($sessionUsage)) {
        echo "<p><strong>🔐 Controllers using sessions:</strong> " . implode(', ', $sessionUsage) . "</p>";
    }
}

/**
 * Analyze admin routes only
 */
function analyzeAdminRoutes($routesFile) {
    if (!file_exists($routesFile)) {
        echo "<p>Routes file not found</p>";
        return;
    }
    
    $content = file_get_contents($routesFile);
    
    // Extract admin routes specifically
    preg_match_all('/\$routes->\w+\s*\(\s*[\'"]([^\'"]*admin[^\'"]*)[\'"]/', $content, $matches);
    
    if (!empty($matches[1])) {
        echo "<p><strong>📍 Admin Routes Found:</strong></p>";
        echo "<div class='code-block'>";
        foreach ($matches[1] as $route) {
            echo "• {$route}<br>";
        }
        echo "</div>";
    } else {
        echo "<p>❌ No admin routes found</p>";
    }
    
    // Look for group routes
    if (strpos($content, 'group(') !== false) {
        echo "<p>✅ Route groups detected (likely used for admin organization)</p>";
    }
}

/**
 * Analyze relevant models
 */
function analyzeRelevantModels($modelsDir) {
    if (!is_dir($modelsDir)) {
        echo "<p>Models directory not found</p>";
        return;
    }
    
    $files = glob($modelsDir . '/*.php');
    
    foreach ($files as $file) {
        $fileName = basename($file, '.php');
        echo "<h5>📊 {$fileName}</h5>";
        
        $content = file_get_contents($file);
        
        // Extract table name if specified
        if (preg_match('/\$table\s*=\s*[\'"]([^\'"]+)[\'"]/', $content, $matches)) {
            echo "<p><strong>Table:</strong> {$matches[1]}</p>";
        }
        
        // Extract methods
        preg_match_all('/public\s+function\s+(\w+)\s*\(([^)]*)\)/', $content, $matches);
        if (!empty($matches[1])) {
            echo "<div class='method-list'>";
            echo "<strong>Methods:</strong> " . implode(', ', array_slice($matches[1], 0, 10));
            if (count($matches[1]) > 10) echo " (+" . (count($matches[1]) - 10) . " more)";
            echo "</div>";
        }
    }
}

?>

        <div class="section">
            <h2>🎯 Analysis Summary</h2>
            <p>This focused analysis provides the essential information needed to design the business portal integration.</p>
            <p><strong>Generated:</strong> <?= date('Y-m-d H:i:s') ?></p>
            <div class="warning">
                <strong>🔒 Remember:</strong> Delete this file after copying the information you need.
            </div>
        </div>
    </div>
</body>
</html>