<?php
/**
 * Enhanced Hours Parser Class - Updated with Pre-check and Proper Empty Hours Handling
 * /public/import/classes/HoursParser.php
 */

if (!defined('IMPORT_ACCESS')) {
    die('Direct access not allowed');
}

class HoursParser {
    private $logger;
    private $db;
    private $importOptions;
    
    public function __construct($database, $logger, $importOptions = []) {
        $this->db = $database;
        $this->logger = $logger;
        $this->importOptions = array_merge([
            'skip_unparseable_hours' => false,
            'strict_quality_mode' => false
        ], $importOptions);
    }
    
    /**
     * Pre-check if hours can be parsed (for skip_unparseable_hours option)
     */
    public function canParseHours($hoursData) {
        $csvHours = $hoursData['working_hours_csv_compatible'] ?? '';
        
        if (empty($csvHours)) {
            return false;
        }
        
        // Perform lightweight parsing test
        $schedules = $this->parseCsvHours($csvHours);
        
        if (empty($schedules)) {
            return false;
        }
        
        // Check if we have at least one valid time slot
        foreach ($schedules as $schedule) {
            $parsed = $this->parseHoursText($schedule);
            if (!$parsed['is_closed'] || ($parsed['slot_1_open'] && $parsed['slot_1_close'])) {
                return true; // Found at least one parseable time slot
            }
        }
        
        return false; // All schedules failed to parse or are closed
    }
    
    /**
     * Parse and import business hours for a store
     */
    public function parseAndImportHours($storeId, $hoursData) {
        $csvHours = $hoursData['working_hours_csv_compatible'] ?? '';
        
        if (empty($csvHours)) {
            $this->logger->logHoursParsing($storeId, false, "No hours data available");
            return $this->handleEmptyHours($storeId);
        }
        
        try {
            // Parse the CSV format
            $schedules = $this->parseCsvHours($csvHours);
            
            if (empty($schedules)) {
                $this->logger->logHoursParsing($storeId, false, "Could not parse hours format");
                return $this->handleEmptyHours($storeId);
            }
            
            // Create hour type and schedules
            $hourTypeId = $this->db->createHourType($storeId);
            
            // Process each day's schedule
            $successfulDays = 0;
            $totalDays = 0;
            
            for ($day = 1; $day <= 7; $day++) {
                $totalDays++;
                
                if (isset($schedules[$day])) {
                    $parsed = $this->parseHoursText($schedules[$day]);
                    $this->db->insertSchedule($storeId, $hourTypeId, $day, $parsed);
                    
                    if (!$parsed['is_closed'] || ($parsed['slot_1_open'] && $parsed['slot_1_close'])) {
                        $successfulDays++;
                    }
                } else {
                    // Day not found in data - mark as closed
                    $closedSchedule = [
                        'is_closed' => 1,
                        'slot_1_open' => null,
                        'slot_1_close' => null,
                        'slot_2_open' => null,
                        'slot_2_close' => null
                    ];
                    $this->db->insertSchedule($storeId, $hourTypeId, $day, $closedSchedule);
                }
            }
            
            // Determine success based on parsing results
            if ($successfulDays === 0) {
                $this->logger->logHoursParsing($storeId, false, "All days failed to parse or marked closed");
                return false;
            }
            
            // Success - log the results
            $hoursType = $this->determineHoursType($successfulDays, $totalDays);
            $this->logger->logHoursParsing($storeId, true, $hoursType);
            
            return true;
            
        } catch (Exception $e) {
            $this->logger->logHoursParsing($storeId, false, $e->getMessage());
            return $this->handleEmptyHours($storeId);
        }
    }
    
    /**
     * Handle cases where no hours can be parsed
     * CRITICAL: Never create false "closed" records
     */
    private function handleEmptyHours($storeId) {
        // DO NOT create any hour_types or schedules records
        // This allows the CRON system to display "Hours not available" 
        // instead of showing false "CLOSED" status
        
        $this->logger->log("NO HOURS CREATED: Store {$storeId} imported without hours data");
        return false; // Return false to indicate hours parsing failed
    }
    
    /**
     * Parse CSV hours format: "Sunday,Open 24 hours|Monday,9:00 AM - 5:00 PM|..."
     */
    private function parseCsvHours($csvHours) {
        $dayEntries = explode('|', $csvHours);
        $schedules = [];
        
        foreach ($dayEntries as $entry) {
            $parts = explode(',', $entry, 2);
            if (count($parts) !== 2) {
                continue;
            }
            
            $dayName = trim($parts[0]);
            $hoursText = trim($parts[1]);
            
            $dayNumber = $this->convertDayToNumber($dayName);
            if (!$dayNumber) {
                continue;
            }
            
            $schedules[$dayNumber] = $hoursText; // Store raw text for parsing
        }
        
        return $schedules;
    }
    
    /**
     * Convert day name to database day number (1=Monday, 7=Sunday)
     */
    private function convertDayToNumber($dayName) {
        return DAY_MAPPINGS[$dayName] ?? null;
    }
    
    /**
     * Parse individual hours text - Enhanced for OutScraper formats
     * Returns proper schedule structure regardless of parsing success
     */
    private function parseHoursText($hoursText) {
        $hoursText = trim($hoursText);
        
        // Handle "Open 24 hours" case
        if (preg_match('/open\s+24\s+hours/i', $hoursText) || preg_match('/24\s*hours/i', $hoursText)) {
            return [
                'is_closed' => 0,
                'slot_1_open' => '00:00:00',
                'slot_1_close' => '23:59:00',
                'slot_2_open' => null,
                'slot_2_close' => null
            ];
        }
        
        // Handle explicit "Closed" case
        if (preg_match('/closed/i', $hoursText)) {
            return [
                'is_closed' => 1,
                'slot_1_open' => null,
                'slot_1_close' => null,
                'slot_2_open' => null,
                'slot_2_close' => null
            ];
        }
        
        // Try standard time ranges first (9:00 AM - 5:00 PM)
        $timeRange = $this->parseStandardTimeRange($hoursText);
        if ($timeRange) {
            return [
                'is_closed' => 0,
                'slot_1_open' => $timeRange['open'],
                'slot_1_close' => $timeRange['close'],
                'slot_2_open' => null,
                'slot_2_close' => null
            ];
        }
        
        // Try comma-separated format (9AM,5PM or 7:30AM,8PM)
        $commaRange = $this->parseCommaSeparatedTime($hoursText);
        if ($commaRange) {
            return [
                'is_closed' => 0,
                'slot_1_open' => $commaRange['open'],
                'slot_1_close' => $commaRange['close'],
                'slot_2_open' => null,
                'slot_2_close' => null
            ];
        }
        
        // Handle multiple time ranges (lunch breaks, etc.)
        $multipleRanges = $this->parseMultipleTimeRanges($hoursText);
        if ($multipleRanges) {
            return $multipleRanges;
        }
        
        // CRITICAL CHANGE: If unparseable, mark as closed but log the issue
        // This is only used when hours data exists but can't be parsed
        $this->logger->log("UNPARSEABLE HOURS: '{$hoursText}' - marking day as closed");
        return [
            'is_closed' => 1,
            'slot_1_open' => null,
            'slot_1_close' => null,
            'slot_2_open' => null,
            'slot_2_close' => null
        ];
    }
    
    /**
     * Parse comma-separated time format (7AM,8PM or 9:30AM,5:30PM)
     */
    private function parseCommaSeparatedTime($text) {
        // Pattern: 7AM,8PM or 9:30AM,5:30PM
        if (preg_match('/(\d{1,2}):?(\d{0,2})(AM|PM)?\s*,\s*(\d{1,2}):?(\d{0,2})(AM|PM)?/i', $text, $matches)) {
            $openHour = intval($matches[1]);
            $openMin = !empty($matches[2]) ? intval($matches[2]) : 0;
            $openAmPm = strtoupper($matches[3] ?? '');
            
            $closeHour = intval($matches[4]);
            $closeMin = !empty($matches[5]) ? intval($matches[5]) : 0;
            $closeAmPm = strtoupper($matches[6] ?? '');
            
            // Default AM/PM logic - morning hours default to AM, evening to PM
            if (empty($openAmPm)) {
                $openAmPm = ($openHour >= 7 && $openHour <= 11) ? 'AM' : 'AM';
            }
            if (empty($closeAmPm)) {
                $closeAmPm = ($closeHour >= 1 && $closeHour <= 11) ? 'PM' : 'PM';
            }
            
            // Validate hours
            if ($openHour < 1 || $openHour > 12 || $closeHour < 1 || $closeHour > 12) {
                return null;
            }
            
            $openTime = $this->convertTo24Hour($openHour, $openMin, $openAmPm);
            $closeTime = $this->convertTo24Hour($closeHour, $closeMin, $closeAmPm);
            
            if ($openTime && $closeTime) {
                return [
                    'open' => $openTime,
                    'close' => $closeTime
                ];
            }
        }
        
        return null;
    }
    
    /**
     * Parse standard time range like "9:00 AM - 5:00 PM"
     */
    private function parseStandardTimeRange($text) {
        // Pattern for time ranges with AM/PM
        if (preg_match('/(\d{1,2}):(\d{2})\s*(AM|PM)?\s*[-–—]\s*(\d{1,2}):(\d{2})\s*(AM|PM)?/i', $text, $matches)) {
            $openHour = intval($matches[1]);
            $openMin = intval($matches[2]);
            $openAmPm = strtoupper($matches[3] ?? '');
            
            $closeHour = intval($matches[4]);
            $closeMin = intval($matches[5]);
            $closeAmPm = strtoupper($matches[6] ?? '');
            
            $openTime = $this->convertTo24Hour($openHour, $openMin, $openAmPm);
            $closeTime = $this->convertTo24Hour($closeHour, $closeMin, $closeAmPm);
            
            if ($openTime && $closeTime) {
                return [
                    'open' => $openTime,
                    'close' => $closeTime
                ];
            }
        }
        
        // Pattern for 24-hour format
        if (preg_match('/(\d{1,2}):(\d{2})\s*[-–—]\s*(\d{1,2}):(\d{2})/', $text, $matches)) {
            $openTime = sprintf('%02d:%02d:00', $matches[1], $matches[2]);
            $closeTime = sprintf('%02d:%02d:00', $matches[3], $matches[4]);
            
            return [
                'open' => $openTime,
                'close' => $closeTime
            ];
        }
        
        return null;
    }
    
    /**
     * Parse multiple time ranges (e.g., "9:00 AM - 12:00 PM, 1:00 PM - 5:00 PM")
     */
    private function parseMultipleTimeRanges($text) {
        // Split by comma, semicolon, or " and "
        $ranges = preg_split('/[,;]|\s+and\s+/i', $text);
        
        if (count($ranges) >= 2) {
            $range1 = $this->parseStandardTimeRange(trim($ranges[0]));
            $range2 = $this->parseStandardTimeRange(trim($ranges[1]));
            
            // Also try comma-separated for each range
            if (!$range1) $range1 = $this->parseCommaSeparatedTime(trim($ranges[0]));
            if (!$range2) $range2 = $this->parseCommaSeparatedTime(trim($ranges[1]));
            
            if ($range1 && $range2) {
                return [
                    'is_closed' => 0,
                    'slot_1_open' => $range1['open'],
                    'slot_1_close' => $range1['close'],
                    'slot_2_open' => $range2['open'],
                    'slot_2_close' => $range2['close']
                ];
            }
        }
        
        return null;
    }
    
    /**
     * Convert 12-hour time to 24-hour format
     */
    private function convertTo24Hour($hour, $minute, $ampm) {
        if ($hour < 1 || $hour > 12 || $minute < 0 || $minute > 59) {
            return null;
        }
        
        if ($ampm === 'PM' && $hour !== 12) {
            $hour += 12;
        } elseif ($ampm === 'AM' && $hour === 12) {
            $hour = 0;
        }
        
        return sprintf('%02d:%02d:00', $hour, $minute);
    }
    
    /**
     * Determine type of hours for logging
     */
    private function determineHoursType($successfulDays, $totalDays) {
        if ($successfulDays === 7) {
            return "Complete week schedule ({$successfulDays}/{$totalDays} days parsed)";
        } elseif ($successfulDays >= 5) {
            return "Most days parsed ({$successfulDays}/{$totalDays} days)";
        } elseif ($successfulDays >= 3) {
            return "Partial schedule ({$successfulDays}/{$totalDays} days parsed)";
        } else {
            return "Limited schedule ({$successfulDays}/{$totalDays} days parsed)";
        }
    }
    
    /**
     * Get sample hours patterns for debugging
     */
    public function analyzePatternsInData($csvData) {
        $patterns = [];
        $dayEntries = explode('|', $csvData);
        
        foreach ($dayEntries as $entry) {
            $parts = explode(',', $entry, 2);
            if (count($parts) === 2) {
                $hoursText = trim($parts[1]);
                if (!isset($patterns[$hoursText])) {
                    $patterns[$hoursText] = 0;
                }
                $patterns[$hoursText]++;
            }
        }
        
        // Sort by frequency
        arsort($patterns);
        
        return array_slice($patterns, 0, 20); // Top 20 patterns
    }
    
    /**
     * Test parsing capabilities on sample data
     */
    public function testParsingCapabilities($sampleHours) {
        $this->logger->log("=== HOURS PARSING TEST ===");
        
        foreach ($sampleHours as $sample) {
            $parsed = $this->parseHoursText($sample);
            $status = $parsed['is_closed'] ? 'CLOSED' : 'OPEN';
            $times = $parsed['is_closed'] ? 'N/A' : $parsed['slot_1_open'] . '-' . $parsed['slot_1_close'];
            
            $this->logger->log("INPUT: '{$sample}' => {$status} ({$times})");
        }
        
        $this->logger->log("=== END PARSING TEST ===");
    }
}
?>