<?php

namespace App\Models;

use CodeIgniter\Model;

class BusinessChangeRequestModel extends Model
{
    protected $table = 'business_change_requests';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $protectFields = true;
    
    protected $allowedFields = [
        'store_id',
        'business_user_id',
        'request_type',
        'field_changes',
        'status',
        'admin_notes',
        'notification_sent',
        'reviewed_at',
        'reviewed_by'
    ];

    // Dates
    protected $useTimestamps = true;
    protected $dateFormat = 'datetime';
    protected $createdField = 'requested_at';
    protected $updatedField = false; // No automatic updated_at field

    // Validation
    protected $validationRules = [
        'store_id' => 'required|max_length[100]',
        'business_user_id' => 'required|integer',
        'request_type' => 'required|in_list[profile_update,hours_update,multiple_fields]',
        'field_changes' => 'required',
        'status' => 'required|in_list[pending,approved,rejected]'
    ];

    protected $validationMessages = [
        'store_id' => [
            'required' => 'Store ID is required.'
        ],
        'business_user_id' => [
            'required' => 'Business user ID is required.',
            'integer' => 'Business user ID must be a valid number.'
        ],
        'request_type' => [
            'required' => 'Request type is required.',
            'in_list' => 'Request type must be profile_update, hours_update, or multiple_fields.'
        ],
        'field_changes' => [
            'required' => 'Field changes data is required.'
        ],
        'status' => [
            'required' => 'Status is required.',
            'in_list' => 'Status must be pending, approved, or rejected.'
        ]
    ];

    protected $skipValidation = false;
    protected $cleanValidationRules = true;

    // Callbacks
    protected $allowCallbacks = true;
    protected $beforeInsert = ['validateFieldChanges'];
    protected $beforeUpdate = ['validateFieldChanges'];

    /**
     * Validate that field_changes is valid JSON
     */
    protected function validateFieldChanges(array $data)
    {
        if (isset($data['data']['field_changes'])) {
            $fieldChanges = $data['data']['field_changes'];
            
            // If it's already an array, encode it
            if (is_array($fieldChanges)) {
                $data['data']['field_changes'] = json_encode($fieldChanges);
            }
            
            // Validate JSON
            if (json_decode($data['data']['field_changes']) === null && json_last_error() !== JSON_ERROR_NONE) {
                throw new \InvalidArgumentException('Field changes must be valid JSON.');
            }
        }
        
        return $data;
    }

    /**
     * Create a new change request
     */
    public function createRequest($storeId, $businessUserId, $requestType, $fieldChanges, $adminNotes = null)
    {
        $data = [
            'store_id' => $storeId,
            'business_user_id' => $businessUserId,
            'request_type' => $requestType,
            'field_changes' => is_array($fieldChanges) ? json_encode($fieldChanges) : $fieldChanges,
            'status' => 'pending',
            'admin_notes' => $adminNotes,
            'notification_sent' => 0
        ];

        return $this->insert($data);
    }

    /**
     * Get all change requests for a specific store
     */
    public function getRequestsByStore($storeId, $limit = null)
    {
        $builder = $this->select('business_change_requests.*, business_users.first_name, business_users.last_name, business_users.username')
                        ->join('business_users', 'business_users.id = business_change_requests.business_user_id', 'left')
                        ->where('business_change_requests.store_id', $storeId)
                        ->orderBy('business_change_requests.requested_at', 'DESC');

        if ($limit) {
            $builder->limit($limit);
        }

        return $builder->findAll();
    }

    /**
     * Get change request by ID and store (security check)
     */
    public function getRequestByIdAndStore($requestId, $storeId)
    {
        return $this->select('business_change_requests.*, business_users.first_name, business_users.last_name, business_users.username, admin.username as admin_username')
                    ->join('business_users', 'business_users.id = business_change_requests.business_user_id', 'left')
                    ->join('admin', 'admin.id = business_change_requests.reviewed_by', 'left')
                    ->where('business_change_requests.id', $requestId)
                    ->where('business_change_requests.store_id', $storeId)
                    ->first();
    }

    /**
     * Get pending change requests count for a store
     */
    public function getPendingRequestsCount($storeId)
    {
        return $this->where('store_id', $storeId)
                    ->where('status', 'pending')
                    ->countAllResults();
    }

    /**
     * Get approved change requests count for a store
     */
    public function getApprovedRequestsCount($storeId)
    {
        return $this->where('store_id', $storeId)
                    ->where('status', 'approved')
                    ->countAllResults();
    }

    /**
     * Get rejected change requests count for a store
     */
    public function getRejectedRequestsCount($storeId)
    {
        return $this->where('store_id', $storeId)
                    ->where('status', 'rejected')
                    ->countAllResults();
    }

    /**
     * Get total change requests count for a store
     */
    public function getTotalRequestsCount($storeId)
    {
        return $this->where('store_id', $storeId)
                    ->countAllResults();
    }

    /**
     * Get last request date for a store
     */
    public function getLastRequestDate($storeId)
    {
        $result = $this->select('requested_at')
                       ->where('store_id', $storeId)
                       ->orderBy('requested_at', 'DESC')
                       ->first();

        return $result ? $result['requested_at'] : null;
    }

    /**
     * Get all pending requests for admin review
     */
    public function getAllPendingRequests($limit = null)
    {
        $builder = $this->select('business_change_requests.*, business_users.first_name, business_users.last_name, business_users.username, store.name as store_name')
                        ->join('business_users', 'business_users.id = business_change_requests.business_user_id', 'left')
                        ->join('store', 'store.store_id = business_change_requests.store_id', 'left')
                        ->where('business_change_requests.status', 'pending')
                        ->orderBy('business_change_requests.requested_at', 'ASC');

        if ($limit) {
            $builder->limit($limit);
        }

        return $builder->findAll();
    }

    /**
     * Get request with full details for admin review
     */
    public function getRequestForReview($requestId)
    {
        return $this->select('business_change_requests.*, business_users.first_name, business_users.last_name, business_users.username, business_users.email, store.name as store_name, store.email as store_email')
                    ->join('business_users', 'business_users.id = business_change_requests.business_user_id', 'left')
                    ->join('store', 'store.store_id = business_change_requests.store_id', 'left')
                    ->where('business_change_requests.id', $requestId)
                    ->first();
    }

    /**
     * Approve a change request
     */
    public function approveRequest($requestId, $adminId, $adminNotes = null)
    {
        $data = [
            'status' => 'approved',
            'reviewed_at' => date('Y-m-d H:i:s'),
            'reviewed_by' => $adminId,
            'admin_notes' => $adminNotes
        ];

        return $this->update($requestId, $data);
    }

    /**
     * Reject a change request
     */
    public function rejectRequest($requestId, $adminId, $adminNotes = null)
    {
        $data = [
            'status' => 'rejected',
            'reviewed_at' => date('Y-m-d H:i:s'),
            'reviewed_by' => $adminId,
            'admin_notes' => $adminNotes
        ];

        return $this->update($requestId, $data);
    }

    /**
     * Mark notification as sent
     */
    public function markNotificationSent($requestId)
    {
        return $this->update($requestId, ['notification_sent' => 1]);
    }

    /**
     * Get field changes as array
     */
    public function getFieldChangesArray($requestId)
    {
        $request = $this->find($requestId);
        
        if (!$request) {
            return null;
        }

        return json_decode($request['field_changes'], true);
    }

    /**
     * Check if there are pending requests for a store
     */
    public function hasPendingRequests($storeId)
    {
        return $this->getPendingRequestsCount($storeId) > 0;
    }

    /**
     * Get recent activity for dashboard
     */
    public function getRecentActivity($storeId, $days = 30, $limit = 10)
    {
        $cutoffDate = date('Y-m-d H:i:s', strtotime("-{$days} days"));
        
        return $this->select('business_change_requests.*, business_users.first_name, business_users.last_name')
                    ->join('business_users', 'business_users.id = business_change_requests.business_user_id', 'left')
                    ->where('business_change_requests.store_id', $storeId)
                    ->where('business_change_requests.requested_at >=', $cutoffDate)
                    ->orderBy('business_change_requests.requested_at', 'DESC')
                    ->limit($limit)
                    ->findAll();
    }

    /**
     * Get requests by status
     */
    public function getRequestsByStatus($status, $storeId = null, $limit = null)
    {
        $builder = $this->select('business_change_requests.*, business_users.first_name, business_users.last_name, business_users.username, store.name as store_name')
                        ->join('business_users', 'business_users.id = business_change_requests.business_user_id', 'left')
                        ->join('store', 'store.store_id = business_change_requests.store_id', 'left')
                        ->where('business_change_requests.status', $status)
                        ->orderBy('business_change_requests.requested_at', 'DESC');

        if ($storeId) {
            $builder->where('business_change_requests.store_id', $storeId);
        }

        if ($limit) {
            $builder->limit($limit);
        }

        return $builder->findAll();
    }

    /**
     * Get statistics for admin dashboard
     */
    public function getAdminStats()
    {
        return [
            'total_requests' => $this->countAll(),
            'pending_requests' => $this->where('status', 'pending')->countAllResults(),
            'approved_requests' => $this->where('status', 'approved')->countAllResults(),
            'rejected_requests' => $this->where('status', 'rejected')->countAllResults(),
            'recent_requests' => $this->where('requested_at >=', date('Y-m-d H:i:s', strtotime('-7 days')))->countAllResults()
        ];
    }
}