<?php

namespace App\Controllers\Cron;

use App\Controllers\BaseController;
use App\Models\StoreModel;

class StoreStatusUpdater extends BaseController
{
    protected $db;
    protected $startTime;

    public function __construct()
    {
        $this->db = \Config\Database::connect();
        $this->startTime = microtime(true);
    }

    /**
     * Main CRON job entry point - run this hourly
     * Usage: php public/index.php cron/store-status-updater/updateAllStatuses
     */
    public function updateAllStatuses()
    {
        $this->logMessage("🚀 Starting store status update...");
        
        // Clear old status records
        $this->db->query("DELETE FROM store_current_status WHERE updated_at < DATE_SUB(NOW(), INTERVAL 2 HOUR)");
        
        // Get all active stores
        $stores = $this->db->query("SELECT store_id, name FROM store WHERE status = 1")->getResultArray();
        $this->logMessage("📊 Processing " . count($stores) . " stores...");
        
        $counters = [
            'total' => 0,
            'open' => 0,
            'closing_soon' => 0,
            'closed' => 0,
            'errors' => 0,
            'services' => []
        ];
        
        foreach ($stores as $store) {
            try {
                $this->updateStoreStatus($store['store_id'], $store['name'], $counters);
                $counters['total']++;
                
                // Progress indicator every 50 stores
                if ($counters['total'] % 50 == 0) {
                    $elapsed = round(microtime(true) - $this->startTime, 2);
                    $this->logMessage("⏱️  Processed {$counters['total']}/{" . count($stores) . "} stores in {$elapsed}s");
                }
                
            } catch (\Exception $e) {
                $counters['errors']++;
                $this->logMessage("❌ Error processing {$store['name']} ({$store['store_id']}): " . $e->getMessage());
            }
        }
        
        // Update summary table
        $this->updateSummaryTable($counters);
        
        $elapsed = round(microtime(true) - $this->startTime, 2);
        $this->logMessage("✅ Completed: {$counters['total']} stores, {$counters['open']} open, {$counters['closing_soon']} closing soon, {$counters['closed']} closed, {$counters['errors']} errors in {$elapsed}s");
        
        return $this->response->setJSON([
            'success' => true,
            'processed' => $counters['total'],
            'open' => $counters['open'],
            'closed' => $counters['closed'],
            'errors' => $counters['errors'],
            'execution_time' => $elapsed
        ]);
    }

    /**
     * Update status for a single store and all its service types
     */
    private function updateStoreStatus($storeId, $storeName, &$counters)
    {
        // Get all service types for this store
        $serviceTypes = $this->db->query("
            SELECT DISTINCT custom_name 
            FROM store_hour_types 
            WHERE store_id = ? AND is_active = 1
            ORDER BY affects_main_status DESC, display_order
        ", [$storeId])->getResultArray();
        
        if (empty($serviceTypes)) {
            // No hour types configured, mark as closed
            $this->insertStatusRecord($storeId, 'main', 'closed', 'Hours not configured', null, null, null, []);
            $counters['closed']++;
            return;
        }
        
        $allServicesStatus = [];
        $openServices = [];
        $mainStatus = 'closed';
        $mainMessage = 'Closed';
        $nextOpen = null;
        
        foreach ($serviceTypes as $serviceType) {
            $serviceName = $serviceType['custom_name'];
            $statusData = $this->calculateServiceStatus($storeId, $serviceName);
            
            // Store individual service status
            $this->insertStatusRecord(
                $storeId, 
                $serviceName, 
                $statusData['status'], 
                $statusData['message'],
                $statusData['opens_at'],
                $statusData['closes_at'],
                $statusData['next_open'],
                $statusData['open_services'] ?? []
            );
            
            $allServicesStatus[] = $statusData;
            
            // Track services breakdown
            if (!isset($counters['services'][$serviceName])) {
                $counters['services'][$serviceName] = ['open' => 0, 'closed' => 0, 'closing_soon' => 0];
            }
            $counters['services'][$serviceName][$statusData['status']]++;
            
            // Determine main store status (if any service affecting main status is open)
            if ($this->affectsMainStatus($storeId, $serviceName)) {
                if ($statusData['status'] === 'open' || $statusData['status'] === 'closing_soon') {
                    $openServices[] = $serviceName;
                    if ($mainStatus === 'closed' || ($mainStatus === 'open' && $statusData['status'] === 'closing_soon')) {
                        $mainStatus = $statusData['status'];
                        $mainMessage = $this->formatMainMessage($statusData, $openServices);
                    }
                }
                
                // Track earliest next opening for main status
                if ($statusData['next_open'] && (!$nextOpen || $statusData['next_open'] < $nextOpen)) {
                    $nextOpen = $statusData['next_open'];
                }
            }
        }
        
        // If no services are open, use next opening time
        if (empty($openServices) && $nextOpen) {
            $mainMessage = $this->formatNextOpenMessage($nextOpen);
        }
        
        // Insert main store status
        $this->insertStatusRecord(
            $storeId, 
            'main', 
            $mainStatus, 
            $mainMessage,
            null, // Main doesn't have specific open/close times
            null,
            $nextOpen,
            $openServices
        );
        
        // Update counters
        $counters[$mainStatus]++;
    }

    /**
     * Calculate status for a specific service type
     */
    private function calculateServiceStatus($storeId, $serviceName)
    {
        // Get business timezone
        $storeData = $this->db->query("
            SELECT s.name, s.store_timezone, tz.name as timezone_name 
            FROM store s 
            LEFT JOIN timezone tz ON s.store_timezone = tz.id 
            WHERE s.store_id = ?
        ", [$storeId])->getRowArray();
        
        $businessTimezone = $storeData['timezone_name'] ?? 'America/New_York';
        
        try {
            $timezone = new \DateTimeZone($businessTimezone);
            $now = new \DateTime('now', $timezone);
        } catch (\Exception $e) {
            $timezone = new \DateTimeZone('America/New_York');
            $now = new \DateTime('now', $timezone);
        }
        
        $dayOfWeek = $now->format('N'); // 1 = Monday, 7 = Sunday
        $currentTime = $now->format('H:i:s');
        $currentDate = $now->format('Y-m-d');

        // Check for exceptions first
        $exception = $this->db->query("
            SELECT * FROM store_schedule_exceptions 
            WHERE store_id = ? AND exception_date = ? 
            ORDER BY hour_type_id IS NULL DESC
            LIMIT 1
        ", [$storeId, $currentDate])->getRowArray();

        if ($exception && $exception['exception_type'] === 'closed') {
            return [
                'status' => 'closed',
                'message' => $exception['message'] ?? 'Holiday closed',
                'opens_at' => null,
                'closes_at' => null,
                'next_open' => $this->getNextOpenTime($storeId, $serviceName, $now),
                'open_services' => []
            ];
        }

        // Get schedule for this service type
        $schedule = $this->db->query("
            SELECT ss.slot_1_open, ss.slot_1_close, ss.slot_2_open, ss.slot_2_close, ss.is_closed
            FROM store_hour_types sht
            JOIN store_schedules ss ON sht.id = ss.hour_type_id
            WHERE sht.store_id = ? AND sht.custom_name = ? AND sht.is_active = 1
            AND ss.day_of_week = ?
            AND (ss.effective_start_date IS NULL OR ss.effective_start_date <= ?)
            AND (ss.effective_end_date IS NULL OR ss.effective_end_date >= ?)
            LIMIT 1
        ", [$storeId, $serviceName, $dayOfWeek, $currentDate, $currentDate])->getRowArray();

        if (!$schedule || $schedule['is_closed']) {
            return [
                'status' => 'closed',
                'message' => 'Closed today',
                'opens_at' => null,
                'closes_at' => null,
                'next_open' => $this->getNextOpenTime($storeId, $serviceName, $now),
                'open_services' => []
            ];
        }

        // Check time slots
        $slots = [
            ['open' => $schedule['slot_1_open'], 'close' => $schedule['slot_1_close']],
            ['open' => $schedule['slot_2_open'], 'close' => $schedule['slot_2_close']]
        ];

        foreach ($slots as $slot) {
            if (!$slot['open'] || !$slot['close']) continue;

            $result = $this->checkTimeSlot($currentTime, $slot['open'], $slot['close']);
            if ($result['is_open']) {
                $status = $result['closing_soon'] ? 'closing_soon' : 'open';
                $message = $result['closing_soon'] 
                    ? "Closing soon" 
                    : "Open until " . date('g:i A', strtotime($slot['close']));

                return [
                    'status' => $status,
                    'message' => $message,
                    'opens_at' => $slot['open'],
                    'closes_at' => $slot['close'],
                    'next_open' => null,
                    'open_services' => [$serviceName]
                ];
            }
        }

        // Not currently open
        return [
            'status' => 'closed',
            'message' => 'Closed',
            'opens_at' => null,
            'closes_at' => null,
            'next_open' => $this->getNextOpenTime($storeId, $serviceName, $now),
            'open_services' => []
        ];
    }

    /**
     * Check if current time falls within a time slot
     */
    private function checkTimeSlot($currentTime, $openTime, $closeTime)
    {
        $currentMinutes = $this->timeToMinutes($currentTime);
        $openMinutes = $this->timeToMinutes($openTime);
        $closeMinutes = $this->timeToMinutes($closeTime);
        
        $isOpen = false;
        $closingSoon = false;

        if ($closeMinutes <= $openMinutes) {
            // Crosses midnight
            if ($currentMinutes >= $openMinutes || $currentMinutes <= $closeMinutes) {
                $isOpen = true;
                if ($currentMinutes >= $openMinutes) {
                    $minutesToMidnight = (24 * 60) - $currentMinutes;
                    if ($minutesToMidnight <= 30) $closingSoon = true;
                } else {
                    if ($closeMinutes - $currentMinutes <= 30) $closingSoon = true;
                }
            }
        } else {
            // Normal same-day schedule
            if ($currentMinutes >= $openMinutes && $currentMinutes <= $closeMinutes) {
                $isOpen = true;
                if ($closeMinutes - $currentMinutes <= 30) $closingSoon = true;
            }
        }

        return ['is_open' => $isOpen, 'closing_soon' => $closingSoon];
    }

    /**
     * Get next opening time for a service
     */
    private function getNextOpenTime($storeId, $serviceName, $now)
    {
        // Check next 7 days
        for ($i = 0; $i < 8; $i++) {
            $checkDate = clone $now;
            $checkDate->add(new \DateInterval("P{$i}D"));
            $dayOfWeek = $checkDate->format('N');
            
            $schedule = $this->db->query("
                SELECT ss.slot_1_open, ss.slot_2_open
                FROM store_hour_types sht
                JOIN store_schedules ss ON sht.id = ss.hour_type_id
                WHERE sht.store_id = ? AND sht.custom_name = ? AND sht.is_active = 1
                AND ss.day_of_week = ? AND ss.is_closed = 0
                ORDER BY ss.slot_1_open
                LIMIT 1
            ", [$storeId, $serviceName, $dayOfWeek])->getRowArray();

            if ($schedule) {
                $openTime = $schedule['slot_1_open'] ?: $schedule['slot_2_open'];
                if ($openTime) {
                    if ($i === 0) {
                        // Today - check if opening time hasn't passed
                        if ($openTime > $now->format('H:i:s')) {
                            return $checkDate->format('Y-m-d') . ' ' . $openTime;
                        }
                    } else {
                        // Future day
                        return $checkDate->format('Y-m-d') . ' ' . $openTime;
                    }
                }
            }
        }

        return null;
    }

    /**
     * Check if service type affects main store status
     */
    private function affectsMainStatus($storeId, $serviceName)
    {
        $result = $this->db->query("
            SELECT affects_main_status 
            FROM store_hour_types 
            WHERE store_id = ? AND custom_name = ? AND is_active = 1
        ", [$storeId, $serviceName])->getRowArray();

        return $result && $result['affects_main_status'] == 1;
    }

    /**
     * Format main status message
     */
    private function formatMainMessage($statusData, $openServices)
    {
        if ($statusData['status'] === 'closing_soon') {
            return count($openServices) > 1 
                ? "Closing soon (" . implode(', ', $openServices) . ")"
                : "Closing soon";
        }

        if ($statusData['status'] === 'open') {
            $message = "Open";
            if (count($openServices) > 1) {
                $message .= " (" . implode(', ', array_slice($openServices, 0, 2));
                if (count($openServices) > 2) {
                    $message .= " +" . (count($openServices) - 2) . " more";
                }
                $message .= ")";
            }
            if ($statusData['closes_at']) {
                $message .= " until " . date('g:i A', strtotime($statusData['closes_at']));
            }
            return $message;
        }

        return 'Closed';
    }

    /**
     * Format next opening message
     */
    private function formatNextOpenMessage($nextOpenDateTime)
    {
        $nextOpen = new \DateTime($nextOpenDateTime);
        $now = new \DateTime();
        
        if ($nextOpen->format('Y-m-d') === $now->format('Y-m-d')) {
            return "Opens today at " . $nextOpen->format('g:i A');
        } elseif ($nextOpen->format('Y-m-d') === $now->add(new \DateInterval('P1D'))->format('Y-m-d')) {
            return "Opens tomorrow at " . $nextOpen->format('g:i A');
        } else {
            return "Opens " . $nextOpen->format('l \a\t g:i A');
        }
    }

    /**
     * Insert or update status record
     */
    private function insertStatusRecord($storeId, $serviceType, $status, $message, $opensAt, $closesAt, $nextOpen, $servicesOpen)
    {
        $this->db->query("
            INSERT INTO store_current_status 
            (store_id, service_type, status, current_message, opens_at, closes_at, next_open_datetime, services_open)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
            status = VALUES(status),
            current_message = VALUES(current_message),
            opens_at = VALUES(opens_at),
            closes_at = VALUES(closes_at),
            next_open_datetime = VALUES(next_open_datetime),
            services_open = VALUES(services_open),
            updated_at = NOW()
        ", [
            $storeId, 
            $serviceType, 
            $status, 
            $message, 
            $opensAt, 
            $closesAt, 
            $nextOpen,
            json_encode($servicesOpen)
        ]);
    }

    /**
     * Update summary table for ultra-fast lookups
     */
    private function updateSummaryTable($counters)
    {
        $this->db->query("
            UPDATE store_status_summary SET
            total_stores = ?,
            total_open = ?,
            total_closing_soon = ?,
            total_closed = ?,
            services_breakdown = ?,
            last_updated = NOW()
        ", [
            $counters['total'],
            $counters['open'],
            $counters['closing_soon'],
            $counters['closed'],
            json_encode($counters['services'])
        ]);
    }

    /**
     * Convert time to minutes
     */
    private function timeToMinutes($timeStr)
    {
        $parts = explode(':', $timeStr);
        return (int)$parts[0] * 60 + (int)$parts[1];
    }

    /**
     * Log message with timestamp
     */
    private function logMessage($message)
    {
        $timestamp = date('Y-m-d H:i:s');
        echo "[$timestamp] $message\n";
        
        // Also log to file if running from CRON
        if (php_sapi_name() === 'cli') {
            error_log("[$timestamp] Store Status CRON: $message");
        }
    }
}