<?php

namespace App\Controllers\Frontend\Ajax;

use App\Controllers\BaseController;
use App\Models\StoreModel;

class StoreDetails extends BaseController
{
    protected $db;

    public function __construct()
    {
        $this->db = \Config\Database::connect();
    }

    public function index()
    {
        $id = $this->request->getPost('id');

        if (!$id) {
            log_message('error', 'StoreDetails: No business ID provided');
            return $this->response->setBody('<p class="text-danger p-4">No business ID provided.</p>');
        }

        // Debug logging
        log_message('debug', 'StoreDetails looking for store with ID: ' . $id);

        $storeModel = new StoreModel();
        
        // Try to find by store_id field (UUID field) - this should be the correct one
        $store = $storeModel->where('store_id', $id)->first();
        
        // If not found, try the primary key as fallback
        if (!$store) {
            log_message('debug', 'Store not found by store_id, trying primary key');
            $store = $storeModel->find($id);
        }
        
        // If still not found, try id field directly
        if (!$store) {
            log_message('debug', 'Store not found by primary key, trying id field');
            $store = $storeModel->where('id', $id)->first();
        }

        if (!$store) {
            log_message('error', 'Store not found for ID: ' . $id);
            return $this->response->setBody('<p class="text-danger p-4">Business not found. ID: ' . esc($id) . '</p>');
        }

        log_message('debug', 'Store found: ' . ($store['name'] ?? 'Unknown'));

        // Ensure we have the store_id for our hour calculations
        $storeIdForHours = $store['store_id'] ?? $store['id'];
        
        // Enhance store data with real-time status and comprehensive hours
        $store = $this->enhanceStoreData($store, $storeIdForHours);

        return view('partials/store_modal', ['store' => $store]);
    }

    /**
     * Enhance store data with real-time status and hours information
     */
    private function enhanceStoreData($store, $storeId)
    {
        // Add real-time status calculation
        $statusData = $this->calculateCurrentStatus($storeId);
        $store['timming']['message'] = $this->formatStatusForModal($statusData);
        $store['is_closed'] = ($statusData['status'] === 'closed') ? '1' : '0';
        
        // Add next opening time for closed businesses
        $store['next_open'] = $this->getNextOpenTime($storeId, $statusData);
        
        // Add holiday information if applicable
        $store['holiday'] = $this->getHolidayInfo($storeId);
        
        // Add comprehensive hours data for modal display
        $store['all_time'] = $this->getFormattedHours($storeId);
        $store['moreHours'] = $this->getAdditionalHourTypes($storeId);
        
        // Add status badge for header
        $store['status_badge'] = $this->getStatusBadge($statusData);
        
        // Add coupon data
        $store['coupon'] = $this->getActiveCoupon($storeId);
        
        // Add jobs data
        $store['active_jobs'] = $this->getActiveJobsForModal($storeId);
        
        return $store;
    }

    /**
     * Get active coupon for the store
     */
    private function getActiveCoupon($storeId)
    {
        try {
            $today = date('Y-m-d');
            
            $coupon = $this->db->query("
                SELECT * FROM business_coupons 
                WHERE store_id = ? 
                AND is_active = 1 
                AND starts_at <= ? 
                AND expires_at >= ?
                ORDER BY created_at DESC
                LIMIT 1
            ", [$storeId, $today, $today])->getRowArray();
            
            if ($coupon) {
                // Format the coupon for display
                return $this->formatCouponForDisplay($coupon);
            }
            
            return null;
        } catch (\Exception $e) {
            log_message('error', 'Error loading coupon for store ' . $storeId . ': ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Format coupon data for modal display
     */
    private function formatCouponForDisplay($coupon)
    {
        // Calculate discount display text
        $discountDisplay = '';
        switch ($coupon['discount_type']) {
            case 'percentage':
                $discountDisplay = intval($coupon['discount_value']) . '% OFF';
                break;
            case 'fixed_amount':
                $discountDisplay = '$' . number_format($coupon['discount_value'], 0) . ' OFF';
                break;
            case 'bogo':
                $discountDisplay = 'BOGO';
                break;
            case 'free_item':
                $discountDisplay = 'FREE ITEM';
                break;
            default:
                $discountDisplay = 'SPECIAL OFFER';
        }
        
        // Calculate days remaining
        $expiresAt = new \DateTime($coupon['expires_at']);
        $today = new \DateTime();
        $daysRemaining = $today->diff($expiresAt)->days;
        
        // If expires today, show as 0 days
        if ($expiresAt->format('Y-m-d') === $today->format('Y-m-d')) {
            $daysRemaining = 0;
        }
        
        $coupon['discount_display'] = $discountDisplay;
        $coupon['days_remaining'] = $daysRemaining;
        
        return $coupon;
    }

    /**
     * Get active jobs for modal display
     */
    private function getActiveJobsForModal($storeId)
    {
        try {
            $today = date('Y-m-d');
            
            $jobs = $this->db->query("
                SELECT id, title, description, employment_type, salary_min, salary_max, salary_type, 
                       contact_method, contact_value, created_at
                FROM business_jobs 
                WHERE store_id = ? 
                AND is_active = 1 
                AND (expires_at IS NULL OR expires_at >= ?)
                ORDER BY created_at DESC
                LIMIT 3
            ", [$storeId, $today])->getResultArray();
            
            // Format each job for display
            $formattedJobs = [];
            foreach ($jobs as $job) {
                $formattedJobs[] = $this->formatJobForModal($job, $storeId);
            }
            
            return $formattedJobs;
        } catch (\Exception $e) {
            log_message('error', 'Error getting active jobs for modal: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Format job data for modal display
     */
    private function formatJobForModal($job, $storeId)
    {
        // Format employment type
        $employmentTypeMap = [
            'full_time' => 'Full Time',
            'part_time' => 'Part Time', 
            'contract' => 'Contract',
            'seasonal' => 'Seasonal'
        ];
        $job['employment_type_display'] = $employmentTypeMap[$job['employment_type']] ?? ucfirst($job['employment_type']);
        
        // Format salary display
        $job['salary_display'] = $this->formatSalaryDisplay($job['salary_min'], $job['salary_max'], $job['salary_type']);
        
        // Format contact method
        $job['contact_method_display'] = $this->formatContactMethodDisplay($job['contact_method'], $job['contact_value'], $storeId);
        
        // Format posted date
        $job['posted_date'] = date('M j, Y', strtotime($job['created_at']));
        
        return $job;
    }

    /**
     * Format salary for display
     */
    private function formatSalaryDisplay($salaryMin, $salaryMax, $salaryType)
    {
        $typeMap = [
            'hourly' => '/hour',
            'monthly' => '/month', 
            'yearly' => '/year'
        ];
        
        $suffix = $typeMap[$salaryType] ?? '';
        
        if ($salaryMax && $salaryMax > $salaryMin) {
            if ($salaryType === 'yearly' && $salaryMin >= 1000) {
                $minDisplay = $salaryMin >= 1000 ? number_format($salaryMin / 1000, 0) . 'K' : number_format($salaryMin, 0);
                $maxDisplay = $salaryMax >= 1000 ? number_format($salaryMax / 1000, 0) . 'K' : number_format($salaryMax, 0);
                return "$" . $minDisplay . "-" . $maxDisplay . $suffix;
            } else {
                return "$" . number_format($salaryMin, 2) . "-" . number_format($salaryMax, 2) . $suffix;
            }
        } else {
            if ($salaryType === 'yearly' && $salaryMin >= 1000) {
                $display = $salaryMin >= 1000 ? number_format($salaryMin / 1000, 0) . 'K' : number_format($salaryMin, 0);
                return "$" . $display . "+" . $suffix;
            } else {
                return "$" . number_format($salaryMin, 2) . "+" . $suffix;
            }
        }
    }

    /**
     * Format contact method for display
     */
    private function formatContactMethodDisplay($contactMethod, $contactValue, $storeId)
    {
        switch ($contactMethod) {
            case 'phone':
                // Get store phone number
                $storeInfo = $this->db->query("SELECT phone FROM store WHERE store_id = ?", [$storeId])->getRowArray();
                return $storeInfo ? "Call: " . $storeInfo['phone'] : "Call us";
                
            case 'email':
                // Get store email
                $storeInfo = $this->db->query("SELECT email FROM store WHERE store_id = ?", [$storeId])->getRowArray();
                return $storeInfo ? "Email: " . $storeInfo['email'] : "Email us";
                
            case 'walk_in':
                return "Visit us in person";
                
            case 'online':
                return $contactValue ? "Apply online" : "Apply online";
                
            default:
                return "Contact us";
        }
    }

    /**
     * Calculate current open/closed status (same as Locator controller)
     */
    private function calculateCurrentStatus($storeId)
    {
        // Get business timezone
        $storeData = $this->db->query("
            SELECT s.name, s.store_timezone, tz.name as timezone_name 
            FROM store s 
            LEFT JOIN timezone tz ON s.store_timezone = tz.id 
            WHERE s.store_id = ?
        ", [$storeId])->getRowArray();
        
        $businessTimezone = $storeData['timezone_name'] ?? 'America/New_York';
        
        try {
            $timezone = new \DateTimeZone($businessTimezone);
            $now = new \DateTime('now', $timezone);
        } catch (\Exception $e) {
            $timezone = new \DateTimeZone('America/New_York');
            $now = new \DateTime('now', $timezone);
        }
        
        $dayOfWeek = $now->format('N'); // 1 = Monday, 7 = Sunday
        $currentTime = $now->format('H:i:s');
        $currentDate = $now->format('Y-m-d');

        // Check for exceptions first
        $exception = $this->db->query("
            SELECT * FROM store_schedule_exceptions 
            WHERE store_id = ? AND exception_date = ? 
            ORDER BY hour_type_id IS NULL DESC
            LIMIT 1
        ", [$storeId, $currentDate])->getRowArray();

        if ($exception && $exception['exception_type'] === 'closed') {
            return [
                'status' => 'closed', 
                'message' => $exception['message'] ?? 'Closed today', 
                'services' => [],
                'is_holiday' => true,
                'holiday_message' => $exception['message'] ?? 'Holiday closed'
            ];
        }

        // Get all hour types that affect main status
        $statusTypes = $this->db->query("
            SELECT sht.id, sht.custom_name,
                   ss.slot_1_open, ss.slot_1_close, ss.slot_2_open, ss.slot_2_close, ss.is_closed
            FROM store_hour_types sht
            JOIN store_schedules ss ON sht.id = ss.hour_type_id
            WHERE sht.store_id = ? AND sht.affects_main_status = 1 AND sht.is_active = 1
            AND ss.day_of_week = ?
            AND (ss.effective_start_date IS NULL OR ss.effective_start_date <= ?)
            AND (ss.effective_end_date IS NULL OR ss.effective_end_date >= ?)
        ", [$storeId, $dayOfWeek, $currentDate, $currentDate])->getResultArray();

        $openServices = [];
        $closingSoon = [];
        $nextCloseTime = null;

        foreach ($statusTypes as $type) {
            if ($type['is_closed']) continue;

            $isOpen = false;
            $closingSoonCheck = false;
            $currentCloseTime = null;

            // Check slot 1 with midnight handling
            if ($type['slot_1_open'] && $type['slot_1_close']) {
                $result = $this->checkTimeSlot($currentTime, $type['slot_1_open'], $type['slot_1_close']);
                if ($result['is_open']) {
                    $isOpen = true;
                    $closingSoonCheck = $result['closing_soon'];
                    $currentCloseTime = $type['slot_1_close'];
                }
            }

            // Check slot 2 if not already open
            if (!$isOpen && $type['slot_2_open'] && $type['slot_2_close']) {
                $result = $this->checkTimeSlot($currentTime, $type['slot_2_open'], $type['slot_2_close']);
                if ($result['is_open']) {
                    $isOpen = true;
                    $closingSoonCheck = $result['closing_soon'];
                    $currentCloseTime = $type['slot_2_close'];
                }
            }

            if ($isOpen) {
                $openServices[] = $type['custom_name'];
                if ($closingSoonCheck) {
                    $closingSoon[] = $type['custom_name'];
                }
                
                // Track the earliest close time for "Open until" message
                if ($currentCloseTime && (!$nextCloseTime || $currentCloseTime < $nextCloseTime)) {
                    $nextCloseTime = $currentCloseTime;
                }
            }
        }

        if (empty($openServices)) {
            return ['status' => 'closed', 'services' => [], 'message' => 'Closed', 'is_holiday' => false];
        }

        if (!empty($closingSoon)) {
            return [
                'status' => 'closing_soon', 
                'services' => $openServices, 
                'closing_soon' => $closingSoon,
                'close_time' => $nextCloseTime,
                'is_holiday' => false
            ];
        }

        return [
            'status' => 'open', 
            'services' => $openServices, 
            'close_time' => $nextCloseTime,
            'is_holiday' => false
        ];
    }

    /**
     * Check if current time falls within a time slot (handles midnight crossing)
     */
    private function checkTimeSlot($currentTime, $openTime, $closeTime)
    {
        $currentMinutes = $this->timeToMinutes($currentTime);
        $openMinutes = $this->timeToMinutes($openTime);
        $closeMinutes = $this->timeToMinutes($closeTime);
        
        $isOpen = false;
        $closingSoon = false;

        if ($closeMinutes <= $openMinutes) {
            // Crosses midnight (e.g., 18:00 - 03:00)
            if ($currentMinutes >= $openMinutes || $currentMinutes <= $closeMinutes) {
                $isOpen = true;
                // Check closing soon
                if ($currentMinutes >= $openMinutes) {
                    // Same day - check if close to midnight
                    $minutesToMidnight = (24 * 60) - $currentMinutes;
                    if ($minutesToMidnight <= 30) {
                        $closingSoon = true;
                    }
                } else {
                    // Next day - check if close to actual close time
                    if ($closeMinutes - $currentMinutes <= 30) {
                        $closingSoon = true;
                    }
                }
            }
        } else {
            // Normal same-day schedule
            if ($currentMinutes >= $openMinutes && $currentMinutes <= $closeMinutes) {
                $isOpen = true;
                // Check if closing within 30 minutes
                if ($closeMinutes - $currentMinutes <= 30) {
                    $closingSoon = true;
                }
            }
        }

        return ['is_open' => $isOpen, 'closing_soon' => $closingSoon];
    }

    /**
     * Convert time string to minutes since midnight
     */
    private function timeToMinutes($timeStr)
    {
        $parts = explode(':', $timeStr);
        return (int)$parts[0] * 60 + (int)$parts[1];
    }

    /**
     * Format status data for modal display
     */
    private function formatStatusForModal($statusData)
    {
        $status = $statusData['status'];
        $services = $statusData['services'] ?? [];
        $closeTime = $statusData['close_time'] ?? null;

        switch ($status) {
            case 'open':
                $message = 'Open';
                if (!empty($services)) {
                    $message .= ' (' . implode(', ', $services) . ')';
                }
                if ($closeTime) {
                    $message .= ' until ' . date('g:i A', strtotime($closeTime));
                }
                return $message;

            case 'closing_soon':
                $closingSoon = $statusData['closing_soon'] ?? [];
                if (!empty($closingSoon)) {
                    return 'Closing Soon (' . implode(', ', $closingSoon) . ')';
                }
                return 'Closing Soon';

            case 'closed':
            default:
                return $statusData['message'] ?? 'Closed';
        }
    }

    /**
     * Get holiday information for today
     */
    private function getHolidayInfo($storeId)
    {
        $currentDate = date('Y-m-d');
        
        $holiday = $this->db->query("
            SELECT * FROM store_schedule_exceptions 
            WHERE store_id = ? AND exception_date = ? AND exception_type = 'closed'
            LIMIT 1
        ", [$storeId, $currentDate])->getRowArray();

        if ($holiday) {
            return [
                'is_holiday' => true,
                'message' => $holiday['message'] ?? 'Holiday closed'
            ];
        }

        return ['is_holiday' => false];
    }

    /**
     * Get formatted hours for main hour types (for modal display)
     */
    private function getFormattedHours($storeId)
    {
        $mainHours = $this->db->query("
            SELECT sht.custom_name, ss.day_of_week, ss.slot_1_open, ss.slot_1_close, ss.slot_2_open, ss.slot_2_close, ss.is_closed
            FROM store_hour_types sht
            JOIN store_schedules ss ON sht.id = ss.hour_type_id
            WHERE sht.store_id = ? AND sht.affects_main_status = 1 AND sht.is_active = 1
            ORDER BY ss.day_of_week
        ", [$storeId])->getResultArray();

        $formattedHours = [];
        $dayNames = [
            1 => 'Monday', 2 => 'Tuesday', 3 => 'Wednesday', 4 => 'Thursday',
            5 => 'Friday', 6 => 'Saturday', 7 => 'Sunday'
        ];

        // Group by day and format properly
        $daySchedules = [];
        foreach ($mainHours as $hour) {
            $day = $hour['day_of_week'];
            if (!isset($daySchedules[$day])) {
                $daySchedules[$day] = [
                    'day_name' => $dayNames[$day],
                    'is_closed' => $hour['is_closed'],
                    'slots' => []
                ];
            }

            if (!$hour['is_closed']) {
                if ($hour['slot_1_open'] && $hour['slot_1_close']) {
                    $daySchedules[$day]['slots'][] = [
                        'open' => $this->formatTimeForDisplay($hour['slot_1_open']),
                        'close' => $this->formatTimeForDisplay($hour['slot_1_close'])
                    ];
                }
                if ($hour['slot_2_open'] && $hour['slot_2_close']) {
                    $daySchedules[$day]['slots'][] = [
                        'open' => $this->formatTimeForDisplay($hour['slot_2_open']),
                        'close' => $this->formatTimeForDisplay($hour['slot_2_close'])
                    ];
                }
                $daySchedules[$day]['is_closed'] = false;
            }
        }

        // Convert to format expected by your modal (with proper JSON structure)
        for ($day = 1; $day <= 7; $day++) {
            $times = [];
            
            if (isset($daySchedules[$day]) && !$daySchedules[$day]['is_closed']) {
                foreach ($daySchedules[$day]['slots'] as $slot) {
                    $times[] = [
                        'to' => $this->convertDisplayTimeToOriginal($slot['open']),
                        'from' => $this->convertDisplayTimeToOriginal($slot['close'])
                    ];
                }
            }
            
            $formattedHours[] = [
                'day' => $day,
                'time' => json_encode($times),
                'display_text' => $this->formatDayHoursForDisplay($daySchedules[$day] ?? null, $dayNames[$day])
            ];
        }

        return $formattedHours;
    }

    /**
     * Get additional hour types (non-main status affecting) for modal
     */
    private function getAdditionalHourTypes($storeId)
    {
        $additionalTypes = $this->db->query("
            SELECT sht.custom_name, ss.day_of_week, ss.slot_1_open, ss.slot_1_close, ss.slot_2_open, ss.slot_2_close, ss.is_closed
            FROM store_hour_types sht
            JOIN store_schedules ss ON sht.id = ss.hour_type_id
            WHERE sht.store_id = ? AND sht.affects_main_status = 0 AND sht.is_active = 1
            ORDER BY sht.display_order, ss.day_of_week
        ", [$storeId])->getResultArray();

        $moreHours = [];
        $dayNames = [
            1 => 'Monday', 2 => 'Tuesday', 3 => 'Wednesday', 4 => 'Thursday',
            5 => 'Friday', 6 => 'Saturday', 7 => 'Sunday'
        ];

        foreach ($additionalTypes as $type) {
            $typeName = $type['custom_name'];
            if (!isset($moreHours[$typeName])) {
                $moreHours[$typeName] = [];
            }

            $times = [];
            $displayText = 'Closed';

            if (!$type['is_closed']) {
                $slots = [];
                if ($type['slot_1_open'] && $type['slot_1_close']) {
                    $times[] = [
                        'to' => $type['slot_1_open'],
                        'from' => $type['slot_1_close']
                    ];
                    $slots[] = $this->formatTimeForDisplay($type['slot_1_open']) . ' - ' . $this->formatTimeForDisplay($type['slot_1_close']);
                }
                if ($type['slot_2_open'] && $type['slot_2_close']) {
                    $times[] = [
                        'to' => $type['slot_2_open'],
                        'from' => $type['slot_2_close']
                    ];
                    $slots[] = $this->formatTimeForDisplay($type['slot_2_open']) . ' - ' . $this->formatTimeForDisplay($type['slot_2_close']);
                }
                $displayText = implode(', ', $slots);
            }

            $moreHours[$typeName][$type['day_of_week']] = [
                'time' => $times,
                'display_text' => $displayText,
                'day_name' => $dayNames[$type['day_of_week']]
            ];
        }

        return $moreHours;
    }

    /**
     * Format time for user-friendly display (e.g., "9:00 AM")
     */
    private function formatTimeForDisplay($timeStr)
    {
        if (!$timeStr) return '';
        return date('g:i A', strtotime($timeStr));
    }

    /**
     * Convert display time back to database format
     */
    private function convertDisplayTimeToOriginal($displayTime)
    {
        if (!$displayTime) return '';
        return date('H:i:s', strtotime($displayTime));
    }

    /**
     * Format a full day's hours for display
     */
    private function formatDayHoursForDisplay($dayData, $dayName)
    {
        if (!$dayData || $dayData['is_closed'] || empty($dayData['slots'])) {
            return 'Closed';
        }

        $timeRanges = [];
        foreach ($dayData['slots'] as $slot) {
            $timeRanges[] = $slot['open'] . ' - ' . $slot['close'];
        }

        return implode(', ', $timeRanges);
    }

    /**
     * Generate status badge for modal header
     */
    private function getStatusBadge($statusData)
    {
        $status = $statusData['status'];
        $services = $statusData['services'] ?? [];

        switch ($status) {
            case 'open':
                $badge = 'Open';
                if (!empty($services)) {
                    $badge .= ' (' . implode(', ', array_slice($services, 0, 2)) . ')';
                    if (count($services) > 2) {
                        $badge .= ' +' . (count($services) - 2) . ' more';
                    }
                }
                return $badge;

            case 'closing_soon':
                return 'Closing Soon';

            case 'closed':
            default:
                if ($statusData['is_holiday'] ?? false) {
                    return 'Holiday Closed';
                }
                return 'Closed';
        }
    }

    /**
     * Get next opening time for closed businesses
     */
    private function getNextOpenTime($storeId, $statusData)
    {
        if ($statusData['status'] !== 'closed') {
            return null;
        }

        // Get business timezone
        $storeData = $this->db->query("
            SELECT tz.name as timezone_name 
            FROM store s 
            LEFT JOIN timezone tz ON s.store_timezone = tz.id 
            WHERE s.store_id = ?
        ", [$storeId])->getRowArray();
        
        $businessTimezone = $storeData['timezone_name'] ?? 'America/New_York';
        
        try {
            $timezone = new \DateTimeZone($businessTimezone);
            $now = new \DateTime('now', $timezone);
        } catch (\Exception $e) {
            $timezone = new \DateTimeZone('America/New_York');
            $now = new \DateTime('now', $timezone);
        }

        // Check today and next 7 days for next opening
        for ($i = 0; $i < 8; $i++) {
            $checkDate = clone $now;
            $checkDate->add(new \DateInterval("P{$i}D"));
            $dayOfWeek = $checkDate->format('N');
            
            $nextSchedule = $this->db->query("
                SELECT sht.custom_name, ss.slot_1_open, ss.slot_2_open
                FROM store_hour_types sht
                JOIN store_schedules ss ON sht.id = ss.hour_type_id
                WHERE sht.store_id = ? AND sht.affects_main_status = 1 AND sht.is_active = 1
                AND ss.day_of_week = ? AND ss.is_closed = 0
                AND (ss.slot_1_open IS NOT NULL OR ss.slot_2_open IS NOT NULL)
                ORDER BY ss.slot_1_open, ss.slot_2_open
                LIMIT 1
            ", [$storeId, $dayOfWeek])->getRowArray();

            if ($nextSchedule) {
                $openTime = $nextSchedule['slot_1_open'] ?: $nextSchedule['slot_2_open'];
                
                if ($i === 0) {
                    // Today - check if opening time hasn't passed
                    $currentTime = $now->format('H:i:s');
                    if ($openTime > $currentTime) {
                        $timeDisplay = date('g:i A', strtotime($openTime));
                        $serviceName = $nextSchedule['custom_name'];
                        return "Opens today at {$timeDisplay}" . ($serviceName !== 'Regular Hours' ? " ({$serviceName})" : '');
                    }
                } else {
                    // Future day
                    $dayName = $checkDate->format('l');
                    $timeDisplay = date('g:i A', strtotime($openTime));
                    $serviceName = $nextSchedule['custom_name'];
                    
                    if ($i === 1) {
                        return "Opens tomorrow at {$timeDisplay}" . ($serviceName !== 'Regular Hours' ? " ({$serviceName})" : '');
                    } else {
                        return "Opens {$dayName} at {$timeDisplay}" . ($serviceName !== 'Regular Hours' ? " ({$serviceName})" : '');
                    }
                }
            }
        }

        return 'Hours not available';
    }
}