<?php

namespace App\Models;

use CodeIgniter\Model;

class StoreModel extends Model
{
    protected $table = 'store';
    protected $allowedFields = [
        'name', 'description', 'phone', 'address', 'city', 'state', 'postal_code', 
        'latitude', 'longitude', 'website', 'is_closed', 'close_message', 'status'
    ];

    /**
     * Get stores with timing information for search results
     */
    public function getStoresWithTiming($keyword = '', $location = '', $page = 1, $limit = 12)
    {
        $offset = ($page - 1) * $limit;
        
        $builder = $this->db->table('store s');
        $builder->select('s.*, 
                         GROUP_CONCAT(DISTINCT CONCAT(sth.day, ":", sth.time) ORDER BY sth.day SEPARATOR "||") as timing_data,
                         GROUP_CONCAT(DISTINCT CONCAT(sh.date, ":", sh.message) SEPARATOR "||") as holiday_data');
        
        $builder->join('store_timing_hours_2 sth', 's.id = sth.store_id', 'left');
        $builder->join('store_holiday sh', 's.id = sh.store_id AND sh.date = CURDATE()', 'left');
        
        if ($keyword) {
            $builder->groupStart()
                   ->like('s.name', $keyword)
                   ->orLike('s.description', $keyword)
                   ->groupEnd();
        }
        
        if ($location) {
            $builder->groupStart()
                   ->like('s.city', $location)
                   ->orLike('s.state', $location)
                   ->orLike('s.postal_code', $location)
                   ->groupEnd();
        }
        
        $builder->where('s.status', 1);
        $builder->groupBy('s.id');
        $builder->orderBy('s.name');
        $builder->limit($limit, $offset);
        
        $results = $builder->get()->getResultArray();
        
        return $this->formatStoresWithTiming($results);
    }

    /**
     * Get stores with coordinates and timing for location-based search
     */
    public function getStoresWithCoordinates($lat, $lng, $keyword = '', $category = '', $limit = 12, $offset = 0, $openOnly = false)
    {
        $builder = $this->db->table('store s');
        $builder->select("s.*, 
                         (3959 * acos(cos(radians($lat)) * cos(radians(s.latitude)) * cos(radians(s.longitude) - radians($lng)) + sin(radians($lat)) * sin(radians(s.latitude)))) AS distance,
                         GROUP_CONCAT(DISTINCT CONCAT(sth.day, ':', sth.time) ORDER BY sth.day SEPARATOR '||') as timing_data,
                         GROUP_CONCAT(DISTINCT CONCAT(sh.date, ':', sh.message) SEPARATOR '||') as holiday_data");
        
        $builder->join('store_timing_hours_2 sth', 's.id = sth.store_id', 'left');
        $builder->join('store_holiday sh', 's.id = sh.store_id AND sh.date = CURDATE()', 'left');
        
        if ($keyword) {
            $builder->groupStart()
                   ->like('s.name', $keyword)
                   ->orLike('s.description', $keyword)
                   ->groupEnd();
        }
        
        if ($category) {
            $builder->where('s.category', $category);
        }
        
        $builder->where('s.status', 1);
        $builder->groupBy('s.id');
        $builder->having('distance <=', 100);
        $builder->orderBy('distance', 'ASC');
        $builder->limit($limit, $offset);
        
        $results = $builder->get()->getResultArray();
        $stores = $this->formatStoresWithTiming($results);
        
        // Filter open stores if requested
        if ($openOnly) {
            $stores = array_filter($stores, function($store) {
                return isset($store['computed_status']) && $store['computed_status'] === 'OPEN';
            });
        }
        
        return $stores;
    }

    /**
     * Get single store with full details for modal
     */
    public function getStoreWithFullDetails($id)
    {
        $builder = $this->db->table('store s');
        $builder->select('s.*');
        $builder->where('s.id', $id);
        $store = $builder->get()->getRowArray();
        
        if (!$store) {
            return null;
        }
        
        // Get timing data
        $timingBuilder = $this->db->table('store_timing_hours_2');
        $timingBuilder->select('day, time');
        $timingBuilder->where('store_id', $id);
        $timingBuilder->orderBy('day');
        $timing = $timingBuilder->get()->getResultArray();
        
        // Get holiday data
        $holidayBuilder = $this->db->table('store_holiday');
        $holidayBuilder->select('date, message');
        $holidayBuilder->where('store_id', $id);
        $holidayBuilder->where('date >=', date('Y-m-d'));
        $holiday = $holidayBuilder->get()->getResultArray();
        
        // Get additional hours (drive-thru, etc.)
        $moreHoursBuilder = $this->db->table('store_more_hours smh');
        $moreHoursBuilder->select('smh.day, smh.time, ht.name as service_name');
        $moreHoursBuilder->join('hours_type ht', 'smh.hours_type_id = ht.id');
        $moreHoursBuilder->where('smh.store_id', $id);
        $moreHoursBuilder->orderBy('ht.name, smh.day');
        $moreHours = $moreHoursBuilder->get()->getResultArray();
        
        // Format the data
        $store['all_time'] = $timing;
        $store['holiday'] = $holiday;
        
        // Group additional hours by service type
        $store['moreHours'] = [];
        foreach ($moreHours as $hour) {
            $serviceName = $hour['service_name'];
            if (!isset($store['moreHours'][$serviceName])) {
                $store['moreHours'][$serviceName] = [];
            }
            $store['moreHours'][$serviceName][$hour['day']] = [
                'time' => json_decode($hour['time'], true) ?: []
            ];
        }
        
        // Calculate current status for modal
        $store = $this->calculateStoreStatus($store);
        
        return $store;
    }

    /**
     * Format stores with timing data and calculate status
     */
    private function formatStoresWithTiming($stores)
    {
        foreach ($stores as &$store) {
            // Parse timing data
            $store['timing_hours'] = [];
            if (!empty($store['timing_data'])) {
                $timingPairs = explode('||', $store['timing_data']);
                foreach ($timingPairs as $pair) {
                    if (strpos($pair, ':') !== false) {
                        list($day, $time) = explode(':', $pair, 2);
                        $store['timing_hours'][$day] = json_decode($time, true) ?: [];
                    }
                }
            }
            
            // Parse holiday data
            $store['holiday_info'] = [];
            if (!empty($store['holiday_data'])) {
                $holidayPairs = explode('||', $store['holiday_data']);
                foreach ($holidayPairs as $pair) {
                    if (strpos($pair, ':') !== false) {
                        list($date, $message) = explode(':', $pair, 2);
                        $store['holiday_info'][$date] = $message;
                    }
                }
            }
            
            // Calculate status
            $store = $this->calculateStoreStatus($store);
            
            // Clean up temporary fields
            unset($store['timing_data'], $store['holiday_data']);
        }
        
        return $stores;
    }

    /**
     * Calculate store open/closed status
     */
    private function calculateStoreStatus($store)
    {
        // Check if store is globally closed
        if (isset($store['is_closed']) && $store['is_closed'] == '1') {
            $store['computed_status'] = 'CLOSED';
            $store['status_message'] = $store['close_message'] ?: 'Temporarily closed';
            $store['open_status'] = 'Closed - ' . $store['status_message'];
            return $store;
        }
        
        // Check for today's holiday
        $today = date('Y-m-d');
        if (isset($store['holiday_info'][$today])) {
            $store['computed_status'] = 'CLOSED';
            $store['status_message'] = $store['holiday_info'][$today] ?: 'Holiday closed';
            $store['open_status'] = 'Closed - ' . $store['status_message'];
            return $store;
        }
        
        // Get today's hours
        $currentDay = date('N'); // 1=Monday, 7=Sunday
        $todayHours = $store['timing_hours'][$currentDay] ?? [];
        
        if (empty($todayHours) || empty(array_filter($todayHours, function($slot) {
            return !empty($slot['to']) && !empty($slot['from']);
        }))) {
            $store['computed_status'] = 'CLOSED';
            $store['status_message'] = 'Closed today';
            $store['open_status'] = 'Closed today';
            return $store;
        }
        
        // Check current time against store hours
        $currentTime = time();
        $isOpen = false;
        $nextOpenTime = null;
        $nextCloseTime = null;
        
        foreach ($todayHours as $slot) {
            if (empty($slot['to']) || empty($slot['from'])) continue;
            
            $openTime = strtotime($slot['to']);
            $closeTime = strtotime($slot['from']);
            
            if ($currentTime >= $openTime && $currentTime <= $closeTime) {
                $isOpen = true;
                $nextCloseTime = $closeTime;
                break;
            } elseif ($currentTime < $openTime && (!$nextOpenTime || $openTime < $nextOpenTime)) {
                $nextOpenTime = $openTime;
            }
        }
        
        if ($isOpen) {
            $store['computed_status'] = 'OPEN';
            $minutesUntilClose = ($nextCloseTime - $currentTime) / 60;
            if ($minutesUntilClose <= 60) {
                $store['status_message'] = 'Closes in ' . round($minutesUntilClose) . ' minutes';
                if ($minutesUntilClose <= 30) {
                    $store['computed_status'] = 'CLOSING_SOON';
                }
            } else {
                $store['status_message'] = 'Open until ' . date('g:i A', $nextCloseTime);
            }
            $store['open_status'] = 'Open - ' . $store['status_message'];
        } else {
            $store['computed_status'] = 'CLOSED';
            if ($nextOpenTime) {
                $minutesUntilOpen = ($nextOpenTime - $currentTime) / 60;
                if ($minutesUntilOpen <= 60) {
                    $store['status_message'] = 'Opens in ' . round($minutesUntilOpen) . ' minutes';
                } else {
                    $store['status_message'] = 'Opens at ' . date('g:i A', $nextOpenTime);
                }
            } else {
                // Find next day that's open
                $store['status_message'] = 'Closed today';
            }
            $store['open_status'] = 'Closed - ' . $store['status_message'];
        }
        
        return $store;
    }

    /**
     * Legacy method for backward compatibility
     */
    public function getStores($keyword = '', $location = '', $page = 1)
    {
        return $this->getStoresWithTiming($keyword, $location, $page);
    }
}