<?php

namespace App\Services;

use CodeIgniter\Email\Email;
use App\Models\BusinessUserModel;
use App\Models\StoreModel;
use App\Models\BusinessChangeRequestModel;

class BusinessEmailService
{
    protected $email;
    protected $businessUserModel;
    protected $storeModel;
    protected $changeRequestModel;

    public function __construct()
    {
        $this->email = \Config\Services::email();
        $this->businessUserModel = new BusinessUserModel();
        $this->storeModel = new StoreModel();
        $this->changeRequestModel = new BusinessChangeRequestModel();
    }

    /**
     * Send notification to admin about new change request
     */
    public function notifyAdminNewRequest($requestId)
    {
        try {
            $request = $this->changeRequestModel->getRequestForReview($requestId);
            
            if (!$request) {
                throw new \Exception('Request not found');
            }

            $subject = "New Business Change Request - {$request['store_name']}";
            
            // Prepare email data
            $data = [
                'request' => $request,
                'review_url' => base_url("admin/change-requests/review/{$requestId}"),
                'admin_url' => base_url('admin/change-requests'),
                'field_changes' => json_decode($request['field_changes'], true)
            ];

            // Send HTML email
            $this->email->setTo('support@ringgoldroad.com');
            $this->email->setSubject($subject);
            $this->email->setMessage($this->loadTemplate('admin_new_request', $data));
            $this->email->setAltMessage($this->loadTemplate('admin_new_request_text', $data));

            if ($this->email->send()) {
                log_message('info', "Admin notification sent for request {$requestId}");
                return true;
            } else {
                throw new \Exception($this->email->printDebugger(['headers']));
            }

        } catch (\Exception $e) {
            log_message('error', "Failed to send admin notification for request {$requestId}: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send notification to business user about request approval
     */
    public function notifyBusinessApproval($requestId, $adminNotes = null)
    {
        try {
            $request = $this->changeRequestModel->getRequestForReview($requestId);
            
            if (!$request) {
                throw new \Exception('Request not found');
            }

            $user = $this->businessUserModel->find($request['business_user_id']);
            
            if (!$user || empty($user['email'])) {
                throw new \Exception('Business user email not found');
            }

            $subject = "Change Request Approved - {$request['store_name']}";
            
            $data = [
                'user' => $user,
                'request' => $request,
                'admin_notes' => $adminNotes,
                'dashboard_url' => base_url('business/dashboard'),
                'field_changes' => json_decode($request['field_changes'], true)
            ];

            $this->email->setTo($user['email']);
            $this->email->setSubject($subject);
            $this->email->setMessage($this->loadTemplate('business_approved', $data));
            $this->email->setAltMessage($this->loadTemplate('business_approved_text', $data));

            if ($this->email->send()) {
                log_message('info', "Approval notification sent to {$user['email']} for request {$requestId}");
                return true;
            } else {
                throw new \Exception($this->email->printDebugger(['headers']));
            }

        } catch (\Exception $e) {
            log_message('error', "Failed to send approval notification for request {$requestId}: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send notification to business user about request rejection
     */
    public function notifyBusinessRejection($requestId, $adminNotes)
    {
        try {
            $request = $this->changeRequestModel->getRequestForReview($requestId);
            
            if (!$request) {
                throw new \Exception('Request not found');
            }

            $user = $this->businessUserModel->find($request['business_user_id']);
            
            if (!$user || empty($user['email'])) {
                throw new \Exception('Business user email not found');
            }

            $subject = "Change Request Needs Revision - {$request['store_name']}";
            
            $data = [
                'user' => $user,
                'request' => $request,
                'admin_notes' => $adminNotes,
                'edit_url' => base_url('business/profile/edit'),
                'dashboard_url' => base_url('business/dashboard'),
                'field_changes' => json_decode($request['field_changes'], true)
            ];

            $this->email->setTo($user['email']);
            $this->email->setSubject($subject);
            $this->email->setMessage($this->loadTemplate('business_rejected', $data));
            $this->email->setAltMessage($this->loadTemplate('business_rejected_text', $data));

            if ($this->email->send()) {
                log_message('info', "Rejection notification sent to {$user['email']} for request {$requestId}");
                return true;
            } else {
                throw new \Exception($this->email->printDebugger(['headers']));
            }

        } catch (\Exception $e) {
            log_message('error', "Failed to send rejection notification for request {$requestId}: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send notification to admin about job changes
     */
    public function notifyAdminJobChange($data)
    {
        try {
            $subject = "Job posting {$data['action']} - {$data['store']['name']}";
            
            // Prepare email data
            $emailData = [
                'action' => $data['action'],
                'store' => $data['store'],
                'job' => $data['job'],
                'user' => $data['user'],
                'admin_url' => $data['admin_url'],
                'timestamp' => $data['timestamp']
            ];

            // Send HTML email
            $this->email->setTo('support@ringgoldroad.com');
            $this->email->setSubject($subject);
            $this->email->setMessage($this->loadTemplate('admin_job_change', $emailData));
            $this->email->setAltMessage($this->loadTemplate('admin_job_change_text', $emailData));

            if ($this->email->send()) {
                log_message('info', "Admin job notification sent for action: {$data['action']}");
                return true;
            } else {
                throw new \Exception($this->email->printDebugger(['headers']));
            }

        } catch (\Exception $e) {
            log_message('error', "Failed to send admin job notification: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send notification to admin about coupon changes
     */
    public function notifyAdminCouponChange($data)
    {
        try {
            $subject = "Coupon {$data['action']} - {$data['store']['name']}";
            
            // Prepare email data
            $emailData = [
                'action' => $data['action'],
                'store' => $data['store'],
                'coupon' => $data['coupon'],
                'user' => $data['user'],
                'admin_url' => $data['admin_url'],
                'timestamp' => $data['timestamp']
            ];

            // Send HTML email
            $this->email->setTo('support@ringgoldroad.com');
            $this->email->setSubject($subject);
            $this->email->setMessage($this->loadTemplate('admin_coupon_change', $emailData));
            $this->email->setAltMessage($this->loadTemplate('admin_coupon_change_text', $emailData));

            if ($this->email->send()) {
                log_message('info', "Admin coupon notification sent for action: {$data['action']}");
                return true;
            } else {
                throw new \Exception($this->email->printDebugger(['headers']));
            }

        } catch (\Exception $e) {
            log_message('error', "Failed to send admin coupon notification: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Send welcome email to new business user
     */
    public function sendWelcomeEmail($userId, $temporaryPassword)
    {
        try {
            $user = $this->businessUserModel->getUserWithStore($userId);
            
            if (!$user || empty($user['email'])) {
                throw new \Exception('Business user email not found');
            }

            $subject = "Welcome to Athens McMinn Business Directory - Your Account Details";
            
            $data = [
                'user' => $user,
                'temporary_password' => $temporaryPassword,
                'login_url' => base_url('business/auth/login'),
                'support_email' => 'support@ringgoldroad.com'
            ];

            $this->email->setTo($user['email']);
            $this->email->setSubject($subject);
            $this->email->setMessage($this->loadTemplate('business_welcome', $data));
            $this->email->setAltMessage($this->loadTemplate('business_welcome_text', $data));

            if ($this->email->send()) {
                log_message('info', "Welcome email sent to {$user['email']}");
                return true;
            } else {
                throw new \Exception($this->email->printDebugger(['headers']));
            }

        } catch (\Exception $e) {
            log_message('error', "Failed to send welcome email to user {$userId}: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Test email configuration
     */
    public function sendTestEmail($toEmail = 'support@ringgoldroad.com')
    {
        try {
            $this->email->setTo($toEmail);
            $this->email->setSubject('Business Directory Email Test');
            $this->email->setMessage('<p>This is a test email from the Athens McMinn Business Directory system.</p><p>If you receive this, email configuration is working correctly!</p><p>Sent at: ' . date('Y-m-d H:i:s') . '</p>');
            $this->email->setAltMessage('This is a test email from the Athens McMinn Business Directory system. If you receive this, email configuration is working correctly! Sent at: ' . date('Y-m-d H:i:s'));

            if ($this->email->send()) {
                return ['success' => true, 'message' => 'Test email sent successfully'];
            } else {
                return ['success' => false, 'message' => $this->email->printDebugger(['headers'])];
            }

        } catch (\Exception $e) {
            return ['success' => false, 'message' => 'Email test failed: ' . $e->getMessage()];
        }
    }

    /**
     * Load email template
     */
    private function loadTemplate($template, $data = [])
    {
        $templatePath = APPPATH . 'Views/emails/' . $template . '.php';
        
        if (!file_exists($templatePath)) {
            // Fallback to basic template if specific one doesn't exist
            return $this->generateBasicTemplate($template, $data);
        }

        // Extract data for template
        extract($data);
        
        // Capture template output
        ob_start();
        include $templatePath;
        return ob_get_clean();
    }

    /**
     * Generate basic email template as fallback
     */
    private function generateBasicTemplate($template, $data)
    {
        switch ($template) {
            case 'admin_new_request':
                return "<h2>New Business Change Request</h2>
                        <p>A new change request has been submitted:</p>
                        <p><strong>Business:</strong> {$data['request']['store_name']}</p>
                        <p><strong>Submitted by:</strong> {$data['request']['first_name']} {$data['request']['last_name']}</p>
                        <p><strong>Request Type:</strong> {$data['request']['request_type']}</p>
                        <p><a href=\"{$data['review_url']}\">Review Request</a></p>";

            case 'admin_new_request_text':
                return "New Business Change Request\n\n" .
                       "Business: {$data['request']['store_name']}\n" .
                       "Submitted by: {$data['request']['first_name']} {$data['request']['last_name']}\n" .
                       "Request Type: {$data['request']['request_type']}\n\n" .
                       "Review at: {$data['review_url']}";

            case 'business_approved':
                return "<h2>Change Request Approved</h2>
                        <p>Hello {$data['user']['first_name']},</p>
                        <p>Your change request for {$data['request']['store_name']} has been approved and the changes are now live.</p>
                        <p><a href=\"{$data['dashboard_url']}\">View Dashboard</a></p>";

            case 'business_approved_text':
                return "Hello {$data['user']['first_name']},\n\n" .
                       "Your change request for {$data['request']['store_name']} has been approved and the changes are now live.\n\n" .
                       "View your dashboard: {$data['dashboard_url']}";

            case 'business_rejected':
                return "<h2>Change Request Needs Revision</h2>
                        <p>Hello {$data['user']['first_name']},</p>
                        <p>Your change request for {$data['request']['store_name']} needs some revisions.</p>
                        <p><strong>Admin Notes:</strong> {$data['admin_notes']}</p>
                        <p><a href=\"{$data['edit_url']}\">Submit Updated Request</a></p>";

            case 'business_rejected_text':
                return "Hello {$data['user']['first_name']},\n\n" .
                       "Your change request for {$data['request']['store_name']} needs some revisions.\n\n" .
                       "Admin Notes: {$data['admin_notes']}\n\n" .
                       "Submit updated request: {$data['edit_url']}";

            case 'admin_job_change':
                $jobTitle = $data['job']['title'] ?? 'Unknown Job';
                $storeName = $data['store']['name'] ?? 'Unknown Store';
                $userEmail = $data['user']['email'] ?? 'Unknown User';
                $action = ucfirst($data['action']);
                
                return "<h2>Job Posting {$action}</h2>
                        <p>A job posting has been {$data['action']}:</p>
                        <p><strong>Business:</strong> {$storeName}</p>
                        <p><strong>Job Title:</strong> {$jobTitle}</p>
                        <p><strong>Action:</strong> {$action}</p>
                        <p><strong>By User:</strong> {$userEmail}</p>
                        <p><strong>Time:</strong> {$data['timestamp']}</p>
                        <p><a href=\"{$data['admin_url']}\">View Admin Jobs</a></p>";

            case 'admin_job_change_text':
                $jobTitle = $data['job']['title'] ?? 'Unknown Job';
                $storeName = $data['store']['name'] ?? 'Unknown Store';
                $userEmail = $data['user']['email'] ?? 'Unknown User';
                $action = ucfirst($data['action']);
                
                return "Job Posting {$action}\n\n" .
                       "A job posting has been {$data['action']}:\n\n" .
                       "Business: {$storeName}\n" .
                       "Job Title: {$jobTitle}\n" .
                       "Action: {$action}\n" .
                       "By User: {$userEmail}\n" .
                       "Time: {$data['timestamp']}\n\n" .
                       "View admin jobs: {$data['admin_url']}";

            case 'admin_coupon_change':
                $couponTitle = $data['coupon']['title'] ?? 'Unknown Coupon';
                $storeName = $data['store']['name'] ?? 'Unknown Store';
                $userEmail = $data['user']['email'] ?? 'Unknown User';
                $action = ucfirst($data['action']);
                
                return "<h2>Coupon {$action}</h2>
                        <p>A coupon has been {$data['action']}:</p>
                        <p><strong>Business:</strong> {$storeName}</p>
                        <p><strong>Coupon Title:</strong> {$couponTitle}</p>
                        <p><strong>Action:</strong> {$action}</p>
                        <p><strong>By User:</strong> {$userEmail}</p>
                        <p><strong>Time:</strong> {$data['timestamp']}</p>
                        <p><a href=\"{$data['admin_url']}\">View Admin Coupons</a></p>";

            case 'admin_coupon_change_text':
                $couponTitle = $data['coupon']['title'] ?? 'Unknown Coupon';
                $storeName = $data['store']['name'] ?? 'Unknown Store';
                $userEmail = $data['user']['email'] ?? 'Unknown User';
                $action = ucfirst($data['action']);
                
                return "Coupon {$action}\n\n" .
                       "A coupon has been {$data['action']}:\n\n" .
                       "Business: {$storeName}\n" .
                       "Coupon Title: {$couponTitle}\n" .
                       "Action: {$action}\n" .
                       "By User: {$userEmail}\n" .
                       "Time: {$data['timestamp']}\n\n" .
                       "View admin coupons: {$data['admin_url']}";

            default:
                return '<p>Email template not found.</p>';
        }
    }
}