<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $title ?></title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            color: #333;
        }

        .admin-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
            padding: 1rem 0;
            position: sticky;
            top: 0;
            z-index: 100;
            box-shadow: 0 2px 20px rgba(0, 0, 0, 0.1);
        }

        .nav-container {
            max-width: 1400px;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0 2rem;
        }

        .nav-brand {
            font-size: 1.5rem;
            font-weight: 700;
            color: #4a5568;
            text-decoration: none;
        }

        .nav-links {
            display: flex;
            gap: 0;
        }

        .nav-links a {
            color: #4a5568;
            text-decoration: none;
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            transition: all 0.3s ease;
            font-weight: 500;
        }

        .nav-links a:hover {
            background: rgba(102, 126, 234, 0.1);
            color: #667eea;
        }

        .nav-links a.active {
            background: #667eea;
            color: white;
            box-shadow: 0 2px 10px rgba(102, 126, 234, 0.3);
        }

        .user-menu {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .user-info {
            font-size: 0.9rem;
            color: #666;
        }

        .logout-btn {
            background: #e53e3e;
            color: white;
            padding: 0.5rem 1rem;
            border-radius: 6px;
            text-decoration: none;
            transition: all 0.3s ease;
        }

        .logout-btn:hover {
            background: #c53030;
            transform: translateY(-1px);
        }

        .main-container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 2rem;
        }

        .page-header {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 2rem;
            margin-bottom: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .page-title {
            font-size: 2.5rem;
            font-weight: 700;
            color: #2d3748;
            margin-bottom: 0.5rem;
        }

        .page-subtitle {
            color: #718096;
            font-size: 1.1rem;
        }

        .assignment-grid {
            display: grid;
            grid-template-columns: 350px 1fr;
            gap: 2rem;
        }

        .categories-panel {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
            height: fit-content;
            position: sticky;
            top: 120px;
        }

        .panel-title {
            font-size: 1.25rem;
            font-weight: 600;
            color: #2d3748;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .category-list {
            list-style: none;
        }

        .category-item {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 0.5rem;
            border: 2px solid #e2e8f0;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .category-item:hover {
            border-color: #667eea;
            background: rgba(102, 126, 234, 0.05);
        }

        .category-item.active {
            border-color: #667eea;
            background: rgba(102, 126, 234, 0.1);
            color: #667eea;
            font-weight: 600;
        }

        .category-icon {
            width: 30px;
            height: 30px;
            background: #f7fafc;
            border-radius: 6px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.9rem;
        }

        .category-info {
            flex: 1;
        }

        .category-name {
            font-weight: 500;
        }

        .category-count {
            font-size: 0.8rem;
            color: #666;
            margin-top: 0.25rem;
        }

        .stores-panel {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 16px;
            padding: 2rem;
            box-shadow: 0 8px 32px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }

        .stores-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 1rem;
        }

        .stores-table th {
            background: #f8fafc;
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            color: #4a5568;
            border-bottom: 2px solid #e2e8f0;
            font-size: 0.9rem;
        }

        .stores-table td {
            padding: 1rem;
            border-bottom: 1px solid #e2e8f0;
            vertical-align: top;
        }

        .stores-table tr:hover {
            background: rgba(102, 126, 234, 0.02);
        }

        .store-name {
            font-weight: 600;
            color: #2d3748;
            font-size: 1.1rem;
            margin-bottom: 0.25rem;
        }

        .store-location {
            color: #718096;
            font-size: 0.9rem;
        }

        .category-badges {
            display: flex;
            flex-wrap: wrap;
            gap: 0.25rem;
        }

        .category-badge {
            background: #bee3f8;
            color: #2b6cb0;
            padding: 0.25rem 0.5rem;
            border-radius: 12px;
            font-size: 0.75rem;
            font-weight: 600;
        }

        .no-categories {
            color: #ed8936;
            font-size: 0.8rem;
            font-style: italic;
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            text-decoration: none;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            transition: all 0.3s ease;
            cursor: pointer;
            font-size: 0.9rem;
        }

        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }

        .btn-primary:hover {
            transform: translateY(-1px);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
        }

        .btn-secondary {
            background: #f7fafc;
            color: #4a5568;
            border: 2px solid #e2e8f0;
        }

        .btn-secondary:hover {
            background: #edf2f7;
            border-color: #cbd5e0;
        }

        .btn-sm {
            padding: 0.5rem 0.75rem;
            font-size: 0.8rem;
            border-radius: 6px;
        }

        .bulk-actions {
            background: #f7fafc;
            border: 2px solid #e2e8f0;
            border-radius: 8px;
            padding: 1rem;
            margin-bottom: 1.5rem;
            display: none;
            align-items: center;
            justify-content: space-between;
        }

        .bulk-actions.show {
            display: flex;
        }

        .bulk-actions-left {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .bulk-actions-right {
            display: flex;
            gap: 0.5rem;
        }

        .empty-state {
            text-align: center;
            padding: 4rem 2rem;
            color: #718096;
        }

        .empty-state i {
            font-size: 3rem;
            margin-bottom: 1rem;
            opacity: 0.5;
        }

        .alert {
            padding: 1rem 1.5rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            border-left: 4px solid;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .alert-success {
            background: #f0fff4;
            border-color: #38a169;
            color: #22543d;
        }

        .alert-error {
            background: #fffaf0;
            border-color: #e53e3e;
            color: #742a2a;
        }

        @media (max-width: 1200px) {
            .assignment-grid {
                grid-template-columns: 1fr;
                gap: 1rem;
            }
            
            .categories-panel {
                position: static;
            }
        }

        @media (max-width: 768px) {
            .main-container {
                padding: 1rem;
            }
            
            .bulk-actions {
                flex-direction: column;
                gap: 1rem;
            }
        }
    </style>
</head>
<body>
    <header class="admin-header">
        <div class="nav-container">
            <a href="/admin/dashboard" class="nav-brand">
                <i class="fas fa-store"></i> Business Directory Admin
            </a>
            
            <nav class="nav-links">
                <a href="/admin/dashboard">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
                <a href="/admin/stores">
                    <i class="fas fa-building"></i> Stores
                </a>
                <a href="/admin/hours">
                    <i class="fas fa-clock"></i> Hours
                </a>
                <a href="/admin/categories" class="active">
                    <i class="fas fa-tags"></i> Categories
                </a>
                <a href="/admin/settings">
                    <i class="fas fa-cog"></i> Settings
                </a>
            </nav>
            
            <div class="user-menu">
                <div class="user-info">
                    <i class="fas fa-user-circle"></i>
                    <?= session()->get('admin_name') ?>
                </div>
                <a href="/admin/logout" class="logout-btn">
                    <i class="fas fa-sign-out-alt"></i> Logout
                </a>
            </div>
        </div>
    </header>

    <main class="main-container">
        <div class="page-header">
            <div style="display: flex; justify-content: space-between; align-items: center;">
                <div>
                    <h1 class="page-title">
                        <i class="fas fa-link"></i> Category Assignments
                    </h1>
                    <p class="page-subtitle">
                        Manage which businesses belong to which categories
                    </p>
                </div>
                <div style="display: flex; gap: 1rem;">
                    <a href="/admin/categories" class="btn btn-secondary">
                        <i class="fas fa-arrow-left"></i> Back to Categories
                    </a>
                    <a href="/admin/categories/create" class="btn btn-primary">
                        <i class="fas fa-plus"></i> Add Category
                    </a>
                </div>
            </div>
        </div>

        <?php if (session()->getFlashdata('success')): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i> <?= session()->getFlashdata('success') ?>
            </div>
        <?php endif; ?>

        <?php if (session()->getFlashdata('error')): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-circle"></i> <?= session()->getFlashdata('error') ?>
            </div>
        <?php endif; ?>

        <div class="assignment-grid">
            <!-- Categories Panel -->
            <div class="categories-panel">
                <h3 class="panel-title">
                    <i class="fas fa-tags"></i> Categories
                </h3>
                
                <div style="margin-bottom: 1rem;">
                    <div class="category-item active" onclick="showAllStores()">
                        <div class="category-icon">
                            <i class="fas fa-building"></i>
                        </div>
                        <div class="category-info">
                            <div class="category-name">All Businesses</div>
                            <div class="category-count"><?= $totalStores ?> total stores</div>
                        </div>
                    </div>
                </div>
                
                <ul class="category-list">
                    <?php foreach ($categories as $category): ?>
                        <li class="category-item" onclick="showCategoryStores('<?= esc($category['category_id']) ?>', '<?= esc($category['title']) ?>')">
                            <div class="category-icon">
                                <i class="<?= esc($category['icon']) ?>"></i>
                            </div>
                            <div class="category-info">
                                <div class="category-name"><?= esc($category['title']) ?></div>
                                <div class="category-count">0 assigned stores</div>
                            </div>
                        </li>
                    <?php endforeach; ?>
                </ul>

                <?php if (empty($categories)): ?>
                    <div style="text-align: center; padding: 2rem; color: #a0aec0;">
                        <i class="fas fa-tags" style="font-size: 2rem; margin-bottom: 1rem;"></i>
                        <p>No categories available.</p>
                        <a href="/admin/categories/create" class="btn btn-primary btn-sm" style="margin-top: 1rem;">
                            <i class="fas fa-plus"></i> Create First Category
                        </a>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Stores Panel -->
            <div class="stores-panel">
                <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1.5rem;">
                    <h3 class="panel-title">
                        <i class="fas fa-building"></i> 
                        <span id="stores-title">All Businesses</span>
                    </h3>
                    <div>
                        <button class="btn btn-primary btn-sm" onclick="showBulkAssignModal()">
                            <i class="fas fa-tags"></i> Bulk Assign Categories
                        </button>
                    </div>
                </div>

                <!-- Bulk Actions -->
                <div class="bulk-actions" id="bulk-actions">
                    <div class="bulk-actions-left">
                        <strong><span id="selected-count">0</span></strong> businesses selected
                    </div>
                    <div class="bulk-actions-right">
                        <button class="btn btn-primary btn-sm" onclick="assignCategoriesToSelected()">
                            <i class="fas fa-tags"></i> Assign Categories
                        </button>
                        <button class="btn btn-secondary btn-sm" onclick="clearSelection()">
                            <i class="fas fa-times"></i> Clear Selection
                        </button>
                    </div>
                </div>

                <div id="stores-content">
                    <?php if (!empty($stores)): ?>
                        <table class="stores-table">
                            <thead>
                                <tr>
                                    <th width="40">
                                        <input type="checkbox" id="select-all" onchange="toggleSelectAll()">
                                    </th>
                                    <th>Business Name</th>
                                    <th>Location</th>
                                    <th>Current Categories</th>
                                    <th width="120">Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($stores as $store): ?>
                                <tr data-store-id="<?= esc($store['store_id']) ?>">
                                    <td>
                                        <input type="checkbox" class="store-checkbox" 
                                               value="<?= esc($store['store_id']) ?>" 
                                               onchange="updateBulkActions()">
                                    </td>
                                    <td>
                                        <div class="store-name"><?= esc($store['name']) ?></div>
                                    </td>
                                    <td>
                                        <div class="store-location">
                                            <i class="fas fa-map-marker-alt"></i>
                                            <?= esc($store['city']) ?>, <?= esc($store['state']) ?>
                                        </div>
                                    </td>
                                    <td>
                                        <?php if ($store['category_count'] > 0): ?>
                                            <div class="category-badges">
                                                <?php 
                                                $categoryList = explode(', ', $store['categories']);
                                                foreach ($categoryList as $category): 
                                                ?>
                                                    <span class="category-badge"><?= esc($category) ?></span>
                                                <?php endforeach; ?>
                                            </div>
                                            <div style="font-size: 0.8rem; color: #666; margin-top: 0.25rem;">
                                                <?= $store['category_count'] ?> categories
                                            </div>
                                        <?php else: ?>
                                            <div class="no-categories">
                                                <i class="fas fa-exclamation-triangle"></i> No categories assigned
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <button class="btn btn-primary btn-sm" 
                                                onclick="assignCategoriesToStore('<?= esc($store['store_id']) ?>', '<?= esc($store['name']) ?>')">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="fas fa-building"></i>
                            <h3>No businesses found</h3>
                            <p>No businesses are currently in the database.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>

    <script>
        let currentCategoryId = null;
        
        function showAllStores() {
            // Remove active class from all category items
            document.querySelectorAll('.category-item').forEach(item => {
                item.classList.remove('active');
            });
            
            // Add active class to "All Businesses"
            document.querySelector('.category-item').classList.add('active');
            
            // Update title
            document.getElementById('stores-title').textContent = 'All Businesses';
            
            currentCategoryId = null;
            
            // Show all stores (you would reload the page or make an AJAX call here)
            // For now, just update the title
        }
        
        function showCategoryStores(categoryId, categoryName) {
            // Remove active class from all category items
            document.querySelectorAll('.category-item').forEach(item => {
                item.classList.remove('active');
            });
            
            // Add active class to clicked category
            event.currentTarget.classList.add('active');
            
            // Update title
            document.getElementById('stores-title').textContent = categoryName + ' Businesses';
            
            currentCategoryId = categoryId;
            
            // Here you would load stores for this category via AJAX
            // For demo purposes, just update the title
        }
        
        function toggleSelectAll() {
            const selectAllCheckbox = document.getElementById('select-all');
            const storeCheckboxes = document.querySelectorAll('.store-checkbox');
            
            storeCheckboxes.forEach(checkbox => {
                checkbox.checked = selectAllCheckbox.checked;
            });
            
            updateBulkActions();
        }
        
        function updateBulkActions() {
            const selectedCheckboxes = document.querySelectorAll('.store-checkbox:checked');
            const count = selectedCheckboxes.length;
            const bulkActions = document.getElementById('bulk-actions');
            
            document.getElementById('selected-count').textContent = count;
            
            if (count > 0) {
                bulkActions.classList.add('show');
            } else {
                bulkActions.classList.remove('show');
            }
        }
        
        function clearSelection() {
            document.getElementById('select-all').checked = false;
            document.querySelectorAll('.store-checkbox').forEach(checkbox => {
                checkbox.checked = false;
            });
            updateBulkActions();
        }
        
        function assignCategoriesToStore(storeId, storeName) {
            alert(`Category assignment for "${storeName}" would open here.\n\nThis would show a modal with checkboxes for all categories.`);
        }
        
        function assignCategoriesToSelected() {
            const selectedIds = Array.from(document.querySelectorAll('.store-checkbox:checked')).map(cb => cb.value);
            
            if (selectedIds.length === 0) {
                alert('Please select businesses first.');
                return;
            }
            
            alert(`Bulk category assignment for ${selectedIds.length} businesses would open here.\n\nThis would show a modal with category checkboxes.`);
        }
        
        function showBulkAssignModal() {
            alert('Bulk assignment modal would open here.\n\nThis would allow selecting multiple stores and assigning categories to all of them at once.');
        }
    </script>
</body>
</html>