<?php

/**
 * CodeIgniter 4 File Usage Analyzer - Web Version
 * 
 * Place this file in your public directory and access via browser
 * Example: http://yoursite.com/analyze_files.php
 */

class FileUsageAnalyzer 
{
    private $projectPath;
    private $appPath;
    private $usedFiles = [];
    private $allFiles = [];
    private $output = '';
    
    public function __construct() 
    {
        // Auto-detect project path from current script location
        $this->projectPath = dirname(__DIR__) . DIRECTORY_SEPARATOR;
        
        // Try common locations for app directory
        if (is_dir($this->projectPath . 'app')) {
            $this->appPath = $this->projectPath . 'app' . DIRECTORY_SEPARATOR;
        } elseif (is_dir(dirname($this->projectPath) . DIRECTORY_SEPARATOR . 'app')) {
            $this->projectPath = dirname($this->projectPath) . DIRECTORY_SEPARATOR;
            $this->appPath = $this->projectPath . 'app' . DIRECTORY_SEPARATOR;
        } else {
            throw new Exception("Could not locate CodeIgniter 4 app directory. Current path: " . $this->projectPath);
        }
        
        $this->log("Analyzing CodeIgniter 4 project at: " . $this->projectPath);
        $this->log("App directory: " . $this->appPath);
    }
    
    private function log($message) 
    {
        $this->output .= htmlspecialchars($message) . "<br>\n";
    }
    
    public function analyze() 
    {
        $this->log("=== CodeIgniter 4 File Usage Analysis ===");
        $this->log("");
        
        try {
            // Get all PHP files
            $this->getAllFiles();
            
            // Analyze routes to see which controllers are registered
            $this->analyzeRoutes();
            
            // Analyze controllers for view usage
            $this->analyzeControllers();
            
            // Analyze views for layout/partial usage
            $this->analyzeViews();
            
            // Generate report
            $this->generateReport();
            
        } catch (Exception $e) {
            $this->log("ERROR: " . $e->getMessage());
        }
        
        return $this->output;
    }
    
    private function getAllFiles() 
    {
        if (!is_dir($this->appPath)) {
            throw new Exception("App directory not found: " . $this->appPath);
        }
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($this->appPath, RecursiveDirectoryIterator::SKIP_DOTS)
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile() && $file->getExtension() === 'php') {
                $relativePath = str_replace($this->appPath, '', $file->getPathname());
                $this->allFiles[] = $relativePath;
            }
        }
        
        $this->log("Found " . count($this->allFiles) . " PHP files");
    }
    
    private function analyzeRoutes() 
    {
        $this->log("Analyzing routes...");
        
        $routesFile = $this->appPath . 'Config' . DIRECTORY_SEPARATOR . 'Routes.php';
        if (file_exists($routesFile)) {
            $content = file_get_contents($routesFile);
            $this->log("  Found Routes.php");
            
            // Look for controller references
            preg_match_all('/([\'"]?)([A-Za-z\\\\]+)(\1)/', $content, $matches);
            
            foreach ($matches[2] as $controller) {
                if (strpos($controller, 'Controller') !== false || strpos($controller, '::') !== false) {
                    $controllerName = str_replace(['App\\Controllers\\', '\\'], ['', DIRECTORY_SEPARATOR], $controller);
                    if (!str_ends_with($controllerName, '.php')) {
                        $controllerName .= '.php';
                    }
                    $this->markFileAsUsed('Controllers', $controllerName);
                    $this->log("  Route references: {$controller}");
                }
            }
            
            // Look for specific route patterns with controllers
            preg_match_all('/\$routes->[a-z]+\([^)]*[\'"]([^\'\"]+::[^\'\"]+)[\'"]/', $content, $routeMatches);
            
            foreach ($routeMatches[1] as $controllerAction) {
                if (strpos($controllerAction, '::') !== false) {
                    $parts = explode('::', $controllerAction);
                    $this->markFileAsUsed('Controllers', $parts[0] . '.php');
                    $this->log("  Route action: {$parts[0]}");
                }
            }
        } else {
            $this->log("  Routes.php not found at: {$routesFile}");
        }
    }
    
    private function analyzeControllers() 
    {
        $this->log("Analyzing controllers...");
        
        $controllerPath = $this->appPath . 'Controllers' . DIRECTORY_SEPARATOR;
        if (is_dir($controllerPath)) {
            $this->analyzeDirectory($controllerPath, 'Controllers');
        } else {
            $this->log("  Controllers directory not found");
        }
    }
    
    private function analyzeViews() 
    {
        $this->log("Analyzing views...");
        
        $viewPath = $this->appPath . 'Views' . DIRECTORY_SEPARATOR;
        if (is_dir($viewPath)) {
            $this->analyzeDirectory($viewPath, 'Views');
        } else {
            $this->log("  Views directory not found");
        }
    }
    
    private function analyzeDirectory($directory, $type) 
    {
        if (!is_dir($directory)) return;
        
        $iterator = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS)
        );
        
        foreach ($iterator as $file) {
            if ($file->isFile() && $file->getExtension() === 'php') {
                $content = file_get_contents($file->getPathname());
                
                // Look for view() calls
                preg_match_all('/view\([\'"]([^\'\"]+)[\'"]/', $content, $viewMatches);
                foreach ($viewMatches[1] as $viewName) {
                    $this->markFileAsUsed('Views', $viewName . '.php');
                    $this->log("  Found view call: {$viewName}");
                }
                
                // Look for extend() calls  
                preg_match_all('/extend\([\'"]([^\'\"]+)[\'"]/', $content, $extendMatches);
                foreach ($extendMatches[1] as $layoutName) {
                    $this->markFileAsUsed('Views', $layoutName . '.php');
                    $this->log("  Found extend call: {$layoutName}");
                }
                
                // Look for include() calls
                preg_match_all('/include\([\'"]([^\'\"]+)[\'"]/', $content, $includeMatches);
                foreach ($includeMatches[1] as $includeName) {
                    $this->markFileAsUsed('Views', $includeName . '.php');
                    $this->log("  Found include call: {$includeName}");
                }
                
                // Look for model usage
                preg_match_all('/new\s+([A-Za-z]+Model)/', $content, $modelMatches);
                foreach ($modelMatches[1] as $modelName) {
                    $this->markFileAsUsed('Models', $modelName . '.php');
                    $this->log("  Found model usage: {$modelName}");
                }
                
                // Mark this file as analyzed
                $relativePath = str_replace($this->appPath, '', $file->getPathname());
                if (!in_array($relativePath, $this->usedFiles)) {
                    $this->usedFiles[] = $relativePath;
                }
            }
        }
    }
    
    private function markFileAsUsed($type, $filename) 
    {
        // Normalize filename to include .php extension if missing
        if (!str_ends_with($filename, '.php')) {
            $filename .= '.php';
        }
        
        $path = $type . DIRECTORY_SEPARATOR . $filename;
        if (!in_array($path, $this->usedFiles)) {
            $this->usedFiles[] = $path;
        }
    }
    
    private function generateReport() 
    {
        $this->log("");
        $this->log("=== ANALYSIS REPORT ===");
        $this->log("");
        
        $this->log("<strong>USED FILES:</strong>");
        $this->log("----------");
        foreach ($this->usedFiles as $file) {
            $this->log("✓ " . $file);
        }
        
        $this->log("");
        $this->log("<strong>POTENTIALLY UNUSED FILES:</strong>");
        $this->log("------------------------");
        
        $unusedCount = 0;
        foreach ($this->allFiles as $file) {
            if (!in_array($file, $this->usedFiles)) {
                $this->log("<span style='color: orange;'>? " . $file . "</span>");
                $unusedCount++;
            }
        }
        
        $this->log("");
        $this->log("=== SUMMARY ===");
        $this->log("Total files: " . count($this->allFiles));
        $this->log("Used files: " . count($this->usedFiles));
        $this->log("Potentially unused: " . $unusedCount);
        
        // Specific analysis for your files
        $this->log("");
        $this->log("=== YOUR SPECIFIC FILES ===");
        $yourFiles = [
            'Controllers' . DIRECTORY_SEPARATOR . 'Home.php' => 'Home (28).php',
            'Views' . DIRECTORY_SEPARATOR . 'layouts' . DIRECTORY_SEPARATOR . 'main.php' => 'main (8).php', 
            'Views' . DIRECTORY_SEPARATOR . 'layouts' . DIRECTORY_SEPARATOR . 'frontend.php' => 'frontend (1).php',
            'Controllers' . DIRECTORY_SEPARATOR . 'Frontend' . DIRECTORY_SEPARATOR . 'Ajax' . DIRECTORY_SEPARATOR . 'Locator.php' => 'Locator (17).php',
            'Views' . DIRECTORY_SEPARATOR . 'home.php' => 'home (27).php'
        ];
        
        foreach ($yourFiles as $expectedPath => $yourFile) {
            $isUsed = in_array($expectedPath, $this->usedFiles);
            $status = $isUsed ? "<span style='color: green;'>✓ USED</span>" : "<span style='color: red;'>? POTENTIALLY UNUSED</span>";
            $this->log("{$status}: {$yourFile} (should be at {$expectedPath})");
        }
    }
}

// Web execution
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>CodeIgniter 4 File Usage Analyzer</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1200px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .output { background: #f8f9fa; padding: 15px; border-radius: 5px; font-family: monospace; font-size: 12px; line-height: 1.4; }
        .header { color: #333; margin-bottom: 20px; }
        .refresh-btn { background: #007bff; color: white; padding: 10px 20px; border: none; border-radius: 5px; cursor: pointer; margin-bottom: 20px; }
        .refresh-btn:hover { background: #0056b3; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>CodeIgniter 4 File Usage Analyzer</h1>
            <p>This tool analyzes your CodeIgniter 4 project to identify which files are actively used vs potentially redundant.</p>
            <button class="refresh-btn" onclick="location.reload()">Refresh Analysis</button>
        </div>
        
        <div class="output">
            <?php
            try {
                $analyzer = new FileUsageAnalyzer();
                echo $analyzer->analyze();
            } catch (Exception $e) {
                echo "<span style='color: red;'>Error: " . htmlspecialchars($e->getMessage()) . "</span><br>";
                echo "Make sure this script is placed in your CodeIgniter 4 public directory.";
            }
            ?>
        </div>
    </div>
</body>
</html>