// ISOLATED KEYWORD SEARCH MODULE
// This script handles ONLY hero keyword search functionality
// Completely isolated from main_script.js to prevent interference

// ===========================================
// KEYWORD SEARCH INITIALIZATION
// ===========================================
document.addEventListener('DOMContentLoaded', function() {
    initializeKeywordSearch();
});

function initializeKeywordSearch() {
    console.log('Initializing isolated keyword search system...');
    
    // Only initialize if keyword search elements exist
    const keywordTab = document.getElementById('keyword-tab');
    const keywordForm = document.getElementById('keyword-search-mode');
    
    if (!keywordTab || !keywordForm) {
        console.log('Keyword search elements not found - skipping initialization');
        return;
    }
    
    setupKeywordTabHandlers();
    setupKeywordFormHandlers();
    
    console.log('Keyword search system initialized successfully');
}

// ===========================================
// KEYWORD TAB MANAGEMENT
// ===========================================
function setupKeywordTabHandlers() {
    const keywordTab = document.getElementById('keyword-tab');
    const zipTab = document.getElementById('zip-tab');
    const addressTab = document.getElementById('address-tab');
    
    if (keywordTab) {
        keywordTab.addEventListener('click', function() {
            switchToKeywordMode();
        });
    }
    
    // Listen for switches away from keyword mode
    if (zipTab) {
        zipTab.addEventListener('click', function() {
            clearKeywordSearch();
        });
    }
    
    if (addressTab) {
        addressTab.addEventListener('click', function() {
            clearKeywordSearch();
        });
    }
}

function switchToKeywordMode() {
    console.log('Switching to keyword search mode...');
    
    const keywordTab = document.getElementById('keyword-tab');
    const zipTab = document.getElementById('zip-tab');
    const addressTab = document.getElementById('address-tab');
    
    const keywordMode = document.getElementById('keyword-search-mode');
    const zipMode = document.getElementById('zip-search-mode');
    const addressMode = document.getElementById('address-search-mode');
    
    const searchModeField = document.getElementById('search_mode');
    
    // Update tab states
    if (keywordTab) keywordTab.classList.add('active');
    if (zipTab) zipTab.classList.remove('active');
    if (addressTab) addressTab.classList.remove('active');
    
    // Update mode displays
    if (keywordMode) keywordMode.style.display = 'block';
    if (zipMode) zipMode.style.display = 'none';
    if (addressMode) addressMode.style.display = 'none';
    
    // Set search mode
    if (searchModeField) searchModeField.value = 'keyword';
    
    // Clear other search inputs
    clearZipInputs();
    clearAddressInput();
    
    // Focus on keyword input
    setTimeout(() => {
        const keywordInput = document.getElementById('keyword_search_input');
        if (keywordInput) keywordInput.focus();
    }, 100);
}

function clearKeywordSearch() {
    const keywordInput = document.getElementById('keyword_search_input');
    if (keywordInput) {
        keywordInput.value = '';
    }
    
    // Clear keyword from hidden field
    const keywordField = document.getElementById('form_keyword');
    if (keywordField) {
        keywordField.value = '';
    }
}

function clearZipInputs() {
    const zipDigits = document.querySelectorAll('.zip-digit');
    zipDigits.forEach(digit => {
        digit.value = '';
        digit.classList.remove('filled');
    });
    
    const zipField = document.getElementById('form_zipcode');
    if (zipField) zipField.value = '';
}

function clearAddressInput() {
    const addressInput = document.getElementById('address_search_input');
    if (addressInput) addressInput.value = '';
}

// ===========================================
// KEYWORD FORM HANDLING
// ===========================================
function setupKeywordFormHandlers() {
    // Handle hero form submission when in keyword mode
    const heroForm = document.getElementById('hero-search-form');
    if (heroForm) {
        heroForm.addEventListener('submit', function(e) {
            const searchMode = document.getElementById('search_mode')?.value;
            if (searchMode === 'keyword') {
                e.preventDefault();
                handleHeroKeywordSearch();
                return false;
            }
        });
    }
    
    // Handle keyword input changes
    const keywordInput = document.getElementById('keyword_search_input');
    if (keywordInput) {
        keywordInput.addEventListener('input', function() {
            updateKeywordField(this.value);
        });
        
        // Handle Enter key in keyword input
        keywordInput.addEventListener('keydown', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                handleHeroKeywordSearch();
            }
        });
    }
    
    // Handle crosshairs button in keyword mode
    const keywordCrosshairs = document.getElementById('keyword-crosshairs-btn');
    if (keywordCrosshairs) {
        keywordCrosshairs.addEventListener('click', function(e) {
            e.preventDefault();
            getLocationForKeywordSearch();
        });
    }
}

function updateKeywordField(keyword) {
    const keywordField = document.getElementById('form_keyword');
    if (keywordField) {
        keywordField.value = keyword.trim();
    }
}

// ===========================================
// KEYWORD SEARCH EXECUTION
// ===========================================
function handleHeroKeywordSearch() {
    const keywordInput = document.getElementById('keyword_search_input');
    const keyword = keywordInput?.value.trim() || '';
    
    console.log('=== HERO KEYWORD SEARCH INITIATED ===', keyword);
    
    if (!keyword) {
        alert('Please enter a business name or keyword to search.');
        if (keywordInput) keywordInput.focus();
        return;
    }
    
    // Update status display
    updateLocationStatus(`Searching for "${keyword}"...`, 'default');
    
    // Set the keyword field (critical for backend)
    const keywordField = document.getElementById('form_keyword');
    if (keywordField) {
        keywordField.value = keyword;
        console.log('Keyword field set to:', keywordField.value);
    }
    
    // Clear other search fields to prevent conflicts
    const zipField = document.getElementById('form_zipcode');
    if (zipField) zipField.value = '';
    
    // Show clear search button
    showClearSearchBtn();
    
    // Mark that a search has been performed
    if (typeof window.hasPerformedSearch !== 'undefined') {
        window.hasPerformedSearch = true;
    }
    
    // Execute the keyword search with location bias
    performKeywordSearchWithLocation(keyword);
}

function performKeywordSearchWithLocation(keyword) {
    console.log('Executing keyword search with location bias...');
    
    // Get current coordinates (may be from previous ZIP/Address searches)
    const latitude = document.getElementById('form_latitude')?.value || null;
    const longitude = document.getElementById('form_longitude')?.value || null;
    
    const hasCoords = latitude && longitude && latitude !== '0' && longitude !== '0';
    
    console.log('Coordinates available:', hasCoords ? `${latitude}, ${longitude}` : 'No coordinates');
    
    // Prepare search data (mirrors sticky header successful approach)
    const searchData = {
        keyword: keyword.trim(),
        location: '',
        category: document.getElementById('selected_category')?.value || '',
        coords: hasCoords ? { lat: parseFloat(latitude), lng: parseFloat(longitude) } : { lat: null, lng: null },
        page: 1,
        open_only: document.getElementById('openOnlyToggle')?.checked || false,
        search_mode: 'keyword'
    };
    
    console.log('Keyword search payload:', searchData);
    
    // Make the API call
    makeKeywordSearchRequest(searchData);
}

function makeKeywordSearchRequest(searchData) {
    // Get the API URL from global config
    const ajaxLocatorUrl = window.siteConfig?.ajaxLocatorUrl || '/ajax/locator';
    
    console.log('Making keyword search API call to:', ajaxLocatorUrl);
    
    // Show loader
    showLoader();
    clearResults();
    
    fetch(ajaxLocatorUrl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(searchData)
    })
    .then(response => response.text())
    .then(html => {
        hideLoader();
        
        console.log('Keyword search response received');
        
        if (html && html.trim()) {
            // Parse metadata
            const metadata = parseMetadata(html) || {};
            
            console.log('Keyword search results:', {
                resultsCount: metadata.results_count || 0,
                hasMore: metadata.has_more_results || false,
                keywordSearch: metadata.keyword_search || ''
            });
            
            // Display results
            const resultsContainer = document.getElementById('store_results');
            if (resultsContainer) {
                resultsContainer.innerHTML = html;
                
                // Enforce mobile constraints if needed
                if (window.innerWidth <= 768) {
                    const newCards = resultsContainer.querySelectorAll('.appstore-card, .stores_list_modal');
                    newCards.forEach(card => {
                        card.style.maxWidth = '100%';
                        card.style.overflow = 'hidden';
                    });
                }
            }
            
            // Update results count display
            updateResultsCount(metadata.displayed_count || 0, searchData.open_only);
            
            // Setup modal handlers for new results
            setupModalHandlers();
            
            // Show filter toggle
            const filterToggle = document.getElementById('filterToggleContainer');
            if (filterToggle) filterToggle.style.display = 'block';
            
            // Update location status
            const keyword = searchData.keyword;
            const resultCount = metadata.results_count || 0;
            const hasCoords = searchData.coords.lat && searchData.coords.lng;
            
            if (resultCount > 0) {
                const locationText = hasCoords ? ' near you' : ' nationwide';
                updateLocationStatus(`Found ${resultCount} results for "${keyword}"${locationText}`, 'success');
            } else {
                updateLocationStatus(`No results found for "${keyword}"`, 'warning');
            }
            
        } else {
            showNoResults();
            updateLocationStatus(`No results found for "${searchData.keyword}"`, 'warning');
        }
    })
    .catch(error => {
        console.error('Keyword search API error:', error);
        hideLoader();
        showNoResults();
        updateLocationStatus('Search error occurred. Please try again.', 'error');
    });
}

// ===========================================
// LOCATION SERVICES FOR KEYWORD SEARCH
// ===========================================
function getLocationForKeywordSearch() {
    updateLocationStatus('Getting your location...', 'default');
    
    if (!navigator.geolocation) {
        alert('Geolocation is not supported by this browser.');
        return;
    }
    
    navigator.geolocation.getCurrentPosition(
        function(position) {
            console.log('Location acquired for keyword search:', position.coords.latitude, position.coords.longitude);
            
            // Set coordinates
            document.getElementById('form_latitude').value = position.coords.latitude;
            document.getElementById('form_longitude').value = position.coords.longitude;
            
            // Get human-readable location
            fetchHumanLocation(position.coords.latitude, position.coords.longitude);
            
            // If user has already entered a keyword, perform search with new location
            const keywordInput = document.getElementById('keyword_search_input');
            const keyword = keywordInput?.value.trim();
            
            if (keyword) {
                updateLocationStatus(`Searching for "${keyword}" near you...`, 'default');
                setTimeout(() => {
                    handleHeroKeywordSearch();
                }, 1000);
            } else {
                updateLocationStatus('Location acquired. Enter a keyword to search nearby.', 'success');
            }
        },
        function(error) {
            console.error('Geolocation error:', error);
            updateLocationStatus('Unable to get location. Search will show nationwide results.', 'warning');
        },
        { 
            timeout: 8000, 
            enableHighAccuracy: false,
            maximumAge: 300000 
        }
    );
}

function fetchHumanLocation(lat, lon) {
    fetch(`https://nominatim.openstreetmap.org/reverse?format=jsonv2&lat=${lat}&lon=${lon}`)
        .then(res => res.json())
        .then(data => {
            if (data && data.address) {
                const city = data.address.city || data.address.town || data.address.village || '';
                const state = data.address.state || '';
                const locationName = `${city}${state ? ', ' + state : ''}`;
                
                if (locationName.trim()) {
                    updateLocationStatus(`Location: ${locationName}`, 'success');
                } else {
                    updateLocationStatus('Location acquired', 'success');
                }
            }
        })
        .catch(error => {
            console.log('Location name lookup failed:', error);
            updateLocationStatus('Location acquired', 'success');
        });
}

// ===========================================
// UTILITY FUNCTIONS
// ===========================================
function updateLocationStatus(message, type = 'default') {
    // Try to use the global function if available
    if (typeof window.updateLocationStatus === 'function') {
        window.updateLocationStatus(message, type);
        return;
    }
    
    // Fallback implementation
    const statusEl = document.getElementById('location-status');
    const textEl = document.getElementById('location-status-text');
    
    if (textEl) textEl.textContent = message;
    if (statusEl) {
        statusEl.classList.remove('success', 'warning', 'error');
        if (type !== 'default') statusEl.classList.add(type);
    }
}

function showLoader() {
    const loader = document.getElementById('loader');
    if (loader) loader.style.display = 'block';
}

function hideLoader() {
    const loader = document.getElementById('loader');
    if (loader) loader.style.display = 'none';
}

function clearResults() {
    const resultsContainer = document.getElementById('store_results');
    const noResults = document.getElementById('no_results');
    if (resultsContainer) resultsContainer.innerHTML = '';
    if (noResults) noResults.style.display = 'none';
}

function showNoResults() {
    const noResults = document.getElementById('no_results');
    if (noResults) noResults.style.display = 'block';
}

function showClearSearchBtn() {
    const clearBtn = document.getElementById('clearSearchBtn');
    if (clearBtn) clearBtn.style.display = 'inline-flex';
}

function parseMetadata(html) {
    const metadataMatch = html.match(/<!-- METADATA: (.+?) -->/);
    if (metadataMatch) {
        try {
            return JSON.parse(metadataMatch[1]);
        } catch (e) {
            console.error('Failed to parse metadata:', e);
        }
    }
    return {};
}

function updateResultsCount(displayedCount, isOpenOnly) {
    // Try to use global function if available
    if (typeof window.updateResultsCount === 'function') {
        window.updateResultsCount(displayedCount, isOpenOnly);
        return;
    }
    
    // Fallback implementation
    const resultsCount = document.getElementById('resultsCount');
    if (resultsCount) {
        const suffix = isOpenOnly ? ' open businesses' : ' businesses';
        resultsCount.textContent = displayedCount + suffix;
    }
}

function setupModalHandlers() {
    // Try to use global function if available
    if (typeof window.setupModalHandlers === 'function') {
        window.setupModalHandlers();
        return;
    }
    
    // Fallback implementation
    const storeCards = document.querySelectorAll('.stores_list_modal');
    
    storeCards.forEach(card => {
        const newCard = card.cloneNode(true);
        card.parentNode.replaceChild(newCard, card);
        
        newCard.style.cursor = 'pointer';
        newCard.addEventListener('click', function() {
            const storeId = this.getAttribute('data-val') || this.dataset.val;
            if (storeId && typeof window.loadStoreModal === 'function') {
                window.loadStoreModal(storeId);
            } else {
                console.error('Modal handler not available');
            }
        });
    });
}

// ===========================================
// GLOBAL EXPOSURE FOR HERO INTEGRATION
// ===========================================
// Make keyword functions available globally for hero section
window.switchToKeywordMode = switchToKeywordMode;
window.handleHeroKeywordSearch = handleHeroKeywordSearch;
window.clearKeywordSearch = clearKeywordSearch;

console.log('Keyword search module loaded and ready');