document.addEventListener('DOMContentLoaded', function () {
  document.querySelector('.typed-words').textContent = "Explore Local Favorites";

  function reverseGeocode(lat, lng) {
    fetch(`https://nominatim.openstreetmap.org/reverse?format=jsonv2&lat=${lat}&lon=${lng}`)
      .then(res => res.json())
      .then(data => {
        const location = data?.address?.road || data?.address?.neighbourhood || data?.address?.city || data?.display_name?.split(',').slice(0, 2).join(', ').trim();
        document.getElementById('location-status').textContent = location ? `Showing results near ${location}` : 'Showing nearby businesses';
      })
      .catch(() => {
        document.getElementById('location-status').textContent = "Showing nearby businesses.";
      });
  }

  function getGeolocationAndLoadStores(reset = false) {
    if (navigator.geolocation) {
      navigator.geolocation.getCurrentPosition(function (position) {
        window.lastKnownLat = position.coords.latitude;
        window.lastKnownLng = position.coords.longitude;
        reverseGeocode(window.lastKnownLat, window.lastKnownLng);
        loadMoreStores(reset);
      }, function () {
        document.getElementById('location-status').textContent = "Location unavailable — showing nearby businesses by IP.";
        loadMoreStores(reset);
      });
    } else {
      document.getElementById('location-status').textContent = "Geolocation not supported by this browser.";
      loadMoreStores(reset);
    }
  }

  let page = 1;
  let loading = false;
  let noMoreResults = false;

  function loadMoreStores(reset = false) {
    if (loading || noMoreResults) return;
    loading = true;
    const loader = document.getElementById('loader');
    loader.style.display = 'block';

    const lat = window?.lastKnownLat || 35.4420;
    const lng = window?.lastKnownLng || -84.5927;
    const keyword = document.getElementById('searchInput')?.value || '';
    const categoryAttr = document.querySelector('.nav-link.active')?.getAttribute('data-category') || '';

    const formData = new URLSearchParams();
    formData.append('lat', lat);
    formData.append('lng', lng);
    formData.append('keyword', keyword);
    formData.append('category', categoryAttr);

    fetch('/ajax/locator?page=' + page, {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: formData.toString()
    })
      .then(response => response.text())
      .then(html => {
        const storeList = document.getElementById('store-list');
        const noResults = document.getElementById('no-results-message');
        const cleaned = html.trim();

        if (reset) {
          storeList.innerHTML = '';
          page = 1;
          noResults.style.display = 'none';
          noMoreResults = false;
        }

        if (cleaned.includes('NO_RESULTS')) {
          loader.style.display = 'none';
          storeList.innerHTML = '';
          noResults.style.display = 'block';
          noMoreResults = true;
        } else {
          storeList.insertAdjacentHTML('beforeend', cleaned);
          noResults.style.display = 'none';
          loader.style.display = 'none';
          page++;
        }

        loading = false;
      })
      .catch(error => {
        console.error('Error loading stores:', error);
        loading = false;
        document.getElementById('loader').style.display = 'none';
        document.getElementById('no-results-message').style.display = 'block';
        noMoreResults = true;
      });
  }

  getGeolocationAndLoadStores();

  window.addEventListener('scroll', () => {
    if ((window.innerHeight + window.scrollY) >= document.body.offsetHeight - 300) {
      loadMoreStores();
    }
  });

  document.getElementById('searchBtn').addEventListener('click', function () {
    page = 1;
    noMoreResults = false;
    getGeolocationAndLoadStores(true);
  });

  document.querySelectorAll('#searchTabs .nav-link').forEach(tab => {
    tab.addEventListener('click', function (e) {
      e.preventDefault();

      document.querySelectorAll('#searchTabs .nav-link').forEach(t => t.classList.remove('active'));
      this.classList.add('active');

      document.getElementById('searchInput').value = '';
      document.getElementById('no-results-message').style.display = 'none';
      document.getElementById('store-list').innerHTML = '';
      document.getElementById('loader').style.display = 'none';

      page = 1;
      loading = false;
      noMoreResults = false;

      getGeolocationAndLoadStores(true);
    });
  });

  document.addEventListener('click', function (e) {
    const card = e.target.closest('.stores_list_modal');
    if (card) {
      const storeId = card.getAttribute('data-val');
      const modalContent = document.getElementById('modal-store-content');
      modalContent.innerHTML = '<p class="text-center">Loading...</p>';

      fetch('/ajax/store_details', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'id=' + encodeURIComponent(storeId)
      })
        .then(response => response.text())
        .then(html => {
          modalContent.innerHTML = html;
          const modal = new bootstrap.Modal(document.getElementById('store-info-dialog'));
          modal.show();
        })
        .catch(error => {
          modalContent.innerHTML = '<p class="text-danger text-center">Failed to load store details.</p>';
          console.error('Modal load error:', error);
        });
    }
  });
});
