// loader-logic.js

let page = 1;
let loading = false;
let noMoreResults = false;

function reverseGeocode(lat, lng) {
  fetch(`https://nominatim.openstreetmap.org/reverse?format=jsonv2&lat=${lat}&lon=${lng}`)
    .then(res => res.json())
    .then(data => {
      const location = data?.address?.road || data?.address?.city || data?.display_name?.split(',').slice(0, 2).join(', ').trim();
      document.getElementById('location-status').textContent = location
        ? `Showing results near ${location}`
        : 'Showing nearby businesses';
    })
    .catch(() => {
      document.getElementById('location-status').textContent = "Showing nearby businesses.";
    });
}

function loadMoreStores(reset = false) {
  if (loading || noMoreResults) return;
  loading = true;

  const loader = document.getElementById('loader');
  const noResults = document.getElementById('no-results-message');

  const lat = typeof window.lastKnownLat !== 'undefined' ? window.lastKnownLat : 35.4420;
  const lng = typeof window.lastKnownLng !== 'undefined' ? window.lastKnownLng : -84.5927;
  const keyword = document.getElementById('searchInput').value;
  const category = document.querySelector('.nav-link.active')?.getAttribute('data-category') || '';

  if (reset) {
    document.getElementById('store-list').innerHTML = '';
    loader.style.display = 'block';
    noResults.style.display = 'none';
    page = 1;
    noMoreResults = false;
  }

  const formData = new URLSearchParams();
  formData.append('lat', lat);
  formData.append('lng', lng);
  formData.append('keyword', keyword);
  formData.append('category', category);

  fetch('/ajax/locator?page=' + page, {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body: formData.toString()
  })
    .then(response => response.text())
    .then(html => {
      const cleaned = html.trim();
      const storeList = document.getElementById('store-list');
      const hasValidCards = cleaned.includes('stores_list_modal');

      if (reset) {
        storeList.innerHTML = '';
      }

      if (hasValidCards) {
        storeList.insertAdjacentHTML('beforeend', cleaned);
        loader.style.display = 'none';
        noResults.style.display = 'none';
        page++;
      } else {
        loader.style.display = 'none';
        if (reset) {
          noResults.style.display = 'block';
          noMoreResults = true;
        }
      }

      loading = false;
    })
    .catch(err => {
      loader.style.display = 'none';
      loading = false;
      noMoreResults = true;
      document.getElementById('no-results-message').style.display = 'block';
      console.error(err);
    });
}

document.addEventListener('DOMContentLoaded', function () {
  const searchBtn = document.getElementById('searchBtn');
  const searchTabs = document.querySelectorAll('#searchTabs .nav-link');

  function tryLoadWithLocation() {
    if (typeof window.lastKnownLat !== 'undefined' && typeof window.lastKnownLng !== 'undefined') {
      reverseGeocode(window.lastKnownLat, window.lastKnownLng);
      loadMoreStores(true);
    }
  }

  if (navigator.geolocation) {
    navigator.geolocation.getCurrentPosition(
      position => {
        window.lastKnownLat = position.coords.latitude;
        window.lastKnownLng = position.coords.longitude;
        tryLoadWithLocation();
      },
      () => {
        document.getElementById('location-status').textContent = "Location unavailable — showing nearby businesses by IP.";
        window.lastKnownLat = 35.4420;
        window.lastKnownLng = -84.5927;
        tryLoadWithLocation();
      }
    );
  } else {
    document.getElementById('location-status').textContent = "Geolocation not supported by this browser.";
    window.lastKnownLat = 35.4420;
    window.lastKnownLng = -84.5927;
    tryLoadWithLocation();
  }

  window.addEventListener('scroll', () => {
    if ((window.innerHeight + window.scrollY) >= document.body.offsetHeight - 300) {
      loadMoreStores();
    }
  });

  searchBtn.addEventListener('click', () => {
    page = 1;
    noMoreResults = false;
    loadMoreStores(true);
  });

  searchTabs.forEach(tab => {
    tab.addEventListener('click', function (e) {
      e.preventDefault();
      document.querySelectorAll('#searchTabs .nav-link').forEach(t => t.classList.remove('active'));
      this.classList.add('active');
      document.getElementById('searchInput').value = '';
      document.getElementById('no-results-message').style.display = 'none';
      page = 1;
      noMoreResults = false;
      loadMoreStores(true);
    });
  });

  document.addEventListener('click', function (e) {
    const card = e.target.closest('.stores_list_modal');
    if (card) {
      const storeId = card.getAttribute('data-val');
      const modalContent = document.getElementById('modal-store-content');
      modalContent.innerHTML = '<p class="text-center">Loading...</p>';

      fetch('/ajax/store_details', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: 'id=' + encodeURIComponent(storeId)
      })
        .then(response => response.text())
        .then(html => {
          modalContent.innerHTML = html;
          const modal = new bootstrap.Modal(document.getElementById('store-info-dialog'));
          modal.show();

          const customCloseBtn = document.getElementById('customModalCloseBtn');
          if (customCloseBtn) {
            customCloseBtn.addEventListener('click', function () {
              modal.hide();
            });
          }
        })
        .catch(error => {
          modalContent.innerHTML = '<p class="text-danger text-center">Failed to load store details.</p>';
          console.error('Modal load error:', error);
        });
    }
  });
});
