<?php
/**
 * Import Logger Class
 * /public/import/classes/ImportLogger.php
 */

if (!defined('IMPORT_ACCESS')) {
    die('Direct access not allowed');
}

class ImportLogger {
    private $logFile;
    private $stats;
    private $startTime;
    private $webOutput;
    
    public function __construct($webOutput = false) {
        $this->webOutput = $webOutput;
        $this->startTime = microtime(true);
        $this->logFile = $this->generateLogFileName();
        $this->initializeStats();
        $this->initializeLogFile();
    }
    
    private function generateLogFileName() {
        $timestamp = date('Y-m-d_H-i-s');
        return IMPORT_SETTINGS['log_file_prefix'] . $timestamp . '.log';
    }
    
    private function initializeStats() {
        $this->stats = [
            'processed' => 0, 'imported' => 0, 'skipped' => 0, 'errors' => 0,
            'hours_parsed' => 0, 'hours_failed' => 0, 'timezone_fallbacks' => 0, 'duplicate_stores' => 0
        ];
    }
    
    private function initializeLogFile() {
        $header = str_repeat('=', 60) . "\n";
        $header .= "OutScraper Import Log - " . date('Y-m-d H:i:s') . "\n";
        $header .= "Database: " . DB_CONFIG['database'] . "\n";
        $header .= str_repeat('=', 60) . "\n\n";
        
        file_put_contents($this->logFile, $header);
        $this->log("Import session started");
    }
    
    public function log($message) {
        $timestamp = date('Y-m-d H:i:s');
        $logEntry = "[{$timestamp}] {$message}" . PHP_EOL;
        
        file_put_contents($this->logFile, $logEntry, FILE_APPEND | LOCK_EX);
        
        if ($this->webOutput) {
            echo "<div class='log-entry'>{$logEntry}</div>";
            @ob_flush(); @flush();
        } else {
            echo $logEntry;
        }
    }
    
    public function logError($message, $context = []) {
        $errorMsg = "ERROR: {$message}";
        if (!empty($context)) {
            $errorMsg .= " | Context: " . json_encode($context);
        }
        $this->log($errorMsg);
        $this->incrementStat('errors');
    }
    
    public function logTimezoneFailure($timezone) {
        $this->log("TIMEZONE FALLBACK: '{$timezone}' not found, using America/New_York (153)");
        $this->incrementStat('timezone_fallbacks');
    }
    
    public function logSkipped($reason, $businessName = null) {
        $message = "SKIPPED: {$reason}";
        if ($businessName) {
            $message .= " | Business: {$businessName}";
        }
        $this->log($message);
        $this->incrementStat('skipped');
    }
    
    public function logImport($businessName, $storeId) {
        $this->log("IMPORTED: {$businessName} (Store ID: {$storeId})");
        $this->incrementStat('imported');
    }
    
    public function logHoursParsing($storeId, $success, $details = null) {
        if ($success) {
            $message = "HOURS PARSED: Store {$storeId}";
            if ($details) $message .= " | {$details}";
            $this->incrementStat('hours_parsed');
        } else {
            $message = "HOURS FAILED: Store {$storeId}";
            if ($details) $message .= " | Error: {$details}";
            $this->incrementStat('hours_failed');
        }
        $this->log($message);
    }
    
    public function logProgress() {
        $processed = $this->stats['processed'];
        if ($processed > 0 && $processed % IMPORT_SETTINGS['batch_size'] === 0) {
            $runtime = $this->getElapsedTime();
            $this->log("PROGRESS: {$processed} records processed | Runtime: {$runtime}");
        }
    }
    
    public function incrementStat($statName) {
        if (isset($this->stats[$statName])) {
            $this->stats[$statName]++;
        }
    }
    
    public function getStats() {
        return $this->stats;
    }
    
    private function getElapsedTime() {
        $elapsed = microtime(true) - $this->startTime;
        
        if ($elapsed < 60) {
            return round($elapsed, 2) . ' seconds';
        } elseif ($elapsed < 3600) {
            $minutes = floor($elapsed / 60);
            $seconds = round($elapsed % 60);
            return "{$minutes}m {$seconds}s";
        } else {
            $hours = floor($elapsed / 3600);
            $minutes = floor(($elapsed % 3600) / 60);
            return "{$hours}h {$minutes}m";
        }
    }
    
    public function printFinalStats() {
        $runtime = $this->getElapsedTime();
        
        $summary = "\n" . str_repeat('=', 60) . "\n";
        $summary .= "IMPORT COMPLETED - Runtime: {$runtime}\n";
        $summary .= str_repeat('=', 60) . "\n";
        
        foreach ($this->stats as $key => $value) {
            $label = ucwords(str_replace('_', ' ', $key));
            $summary .= sprintf("%-20s: %d\n", $label, $value);
        }
        
        $summary .= str_repeat('-', 60) . "\n";
        
        $total = $this->stats['processed'];
        if ($total > 0) {
            $successRate = round(($this->stats['imported'] / $total) * 100, 2);
            $summary .= sprintf("Success Rate: %s%% (%d of %d)\n", $successRate, $this->stats['imported'], $total);
        }
        
        $summary .= "Log File: {$this->logFile}\n";
        $summary .= str_repeat('=', 60) . "\n";
        
        file_put_contents($this->logFile, $summary, FILE_APPEND | LOCK_EX);
        
        if ($this->webOutput) {
            echo "<div class='final-stats'><pre>{$summary}</pre></div>";
        } else {
            echo $summary;
        }
    }
    
    public function getLogFile() { return $this->logFile; }
    public function logValidation($field, $value, $issue) {
        $this->log("VALIDATION: Field '{$field}' = '{$value}' | Issue: {$issue}");
    }
    
    public function getWebSummary() {
        $stats = $this->getStats();
        $runtime = $this->getElapsedTime();
        
        return [
            'runtime' => $runtime,
            'stats' => $stats,
            'log_file' => $this->logFile,
            'success_rate' => $stats['processed'] > 0 ? 
                round(($stats['imported'] / $stats['processed']) * 100, 2) : 0
        ];
    }
}
?>