<?php
/**
 * Business "Verification Light" System - Enhanced Contact and Data Validation
 * /public/tools/verification_light.php
 */

define('VALIDATION_ACCESS', true);
require_once '../app/Config/Database.php';

class VerificationLight {
    private $db;
    private $stats;
    private $commonWords = ['and', 'of', 'the', 'for', 'in', 'on', 'at', 'by', 'to', 'with'];
    
    public function __construct() {
        $this->db = \Config\Database::connect();
        $this->stats = [
            'total_checked' => 0,
            'websites' => ['valid' => 0, 'invalid' => 0, 'timeout' => 0],
            'emails' => ['valid' => 0, 'invalid' => 0, 'suspicious' => 0],
            'phones' => ['valid' => 0, 'invalid' => 0, 'formatted' => 0],
            'zip_codes' => ['valid' => 0, 'invalid' => 0, 'empty' => 0],
            'names' => ['too_long' => 0, 'needs_cleanup' => 0, 'valid' => 0]
        ];
        $this->createValidationTables();
    }
    
    /**
     * Main validation runner
     */
    public function validateAllContacts($batchSize = 50) {
        echo "Starting Verification Light process...\n";
        
        $stores = $this->db->query("
            SELECT store_id, name, website, email, phone, postal_code 
            FROM store 
            WHERE status = 1 
            ORDER BY name 
            LIMIT ?
        ", [$batchSize])->getResultArray();
        
        foreach ($stores as $store) {
            $this->validateStore($store);
            $this->stats['total_checked']++;
            
            if ($this->stats['total_checked'] % 10 == 0) {
                $this->printProgress();
            }
        }
        
        $this->printFinalStats();
        $this->generateSummaryReport();
    }
    
    /**
     * Validate individual store contacts and data
     */
    private function validateStore($store) {
        $results = [
            'store_id' => $store['store_id'],
            'website_status' => $this->validateWebsite($store['website']),
            'email_status' => $this->validateEmail($store['email']),
            'phone_status' => $this->validatePhone($store['phone']),
            'zip_status' => $this->validateZipCode($store['postal_code']),
            'name_analysis' => $this->analyzeName($store['name'])
        ];
        
        // Store results and flag issues
        $this->saveValidationResults($results, $store);
        
        echo sprintf("%-30s | %-8s | %-8s | %-8s | %-8s | %-8s\n",
            substr($store['name'], 0, 29),
            $results['website_status'],
            $results['email_status'],
            $results['phone_status'],
            $results['zip_status'],
            $results['name_analysis']['status']
        );
    }
    
    /**
     * Zip code validation
     */
    private function validateZipCode($zip) {
        if (empty($zip)) {
            $this->stats['zip_codes']['empty']++;
            return 'empty';
        }
        
        // Clean zip code
        $zip = preg_replace('/\D/', '', $zip);
        
        if (strlen($zip) == 5) {
            $this->stats['zip_codes']['valid']++;
            return 'valid';
        } elseif (strlen($zip) == 9) {
            $this->stats['zip_codes']['valid']++;
            return 'valid_plus4';
        } else {
            $this->stats['zip_codes']['invalid']++;
            return 'invalid';
        }
    }
    
    /**
     * Business name analysis and suggestions
     */
    private function analyzeName($name) {
        $issues = [];
        $suggestions = [];
        
        // Count words
        $words = explode(' ', trim($name));
        $wordCount = count($words);
        
        if ($wordCount > 5) {
            $issues[] = 'too_many_words';
            $this->stats['names']['too_long']++;
        }
        
        // Check for LLC/Inc cleanup needed
        $cleanName = $name;
        if (preg_match('/\s*,?\s*(LLC|Inc\.?|Corporation|Corp\.?)$/i', $name)) {
            $issues[] = 'has_legal_suffix';
            $cleanName = preg_replace('/\s*,?\s*(LLC|Inc\.?|Corporation|Corp\.?)$/i', '', $name);
            $suggestions['remove_suffix'] = trim($cleanName);
        }
        
        // Check for capitalization issues
        $properCase = $this->toProperBusinessCase($cleanName);
        if ($properCase !== $name && $properCase !== $cleanName) {
            $issues[] = 'capitalization';
            $suggestions['proper_case'] = $properCase;
        }
        
        // Check for strange characters
        if (preg_match('/[^a-zA-Z0-9\s&\-\'\.\,\(\)]/', $name)) {
            $issues[] = 'strange_characters';
            $cleanChars = preg_replace('/[^a-zA-Z0-9\s&\-\'\.\,\(\)]/', '', $name);
            $suggestions['clean_characters'] = $cleanChars;
        }
        
        if (empty($issues)) {
            $this->stats['names']['valid']++;
            return ['status' => 'valid', 'issues' => [], 'suggestions' => []];
        } else {
            $this->stats['names']['needs_cleanup']++;
            return ['status' => 'flagged', 'issues' => $issues, 'suggestions' => $suggestions];
        }
    }
    
    /**
     * Convert to proper business case
     */
    private function toProperBusinessCase($name) {
        $words = explode(' ', trim($name));
        $result = [];
        
        foreach ($words as $i => $word) {
            $lowerWord = strtolower($word);
            
            // Always capitalize first and last word
            if ($i === 0 || $i === count($words) - 1) {
                $result[] = ucfirst($word);
            } 
            // Keep common words lowercase unless they're first/last
            elseif (in_array($lowerWord, $this->commonWords)) {
                $result[] = $lowerWord;
            }
            // Handle acronyms (all caps, 2-4 letters)
            elseif (strlen($word) <= 4 && strtoupper($word) === $word) {
                $result[] = strtoupper($word);
            }
            // Regular capitalization
            else {
                $result[] = ucfirst(strtolower($word));
            }
        }
        
        return implode(' ', $result);
    }
    
    /**
     * Website validation (same as before)
     */
    private function validateWebsite($url) {
        if (empty($url)) return 'empty';
        
        $url = trim($url);
        if (!preg_match('/^https?:\/\//', $url)) {
            $url = 'http://' . $url;
        }
        
        $domain = parse_url($url, PHP_URL_HOST);
        if (!$domain) {
            $this->stats['websites']['invalid']++;
            return 'invalid';
        }
        
        if (!checkdnsrr($domain, 'A')) {
            $this->stats['websites']['invalid']++;
            return 'no_dns';
        }
        
        $context = stream_context_create([
            'http' => [
                'timeout' => 10,
                'user_agent' => 'Business Directory Validator/1.0',
                'follow_location' => true
            ]
        ]);
        
        $response = @file_get_contents($url, false, $context);
        
        if ($response === false) {
            $error = error_get_last();
            if (strpos($error['message'], 'timed out') !== false) {
                $this->stats['websites']['timeout']++;
                return 'timeout';
            } else {
                $this->stats['websites']['invalid']++;
                return 'unreachable';
            }
        }
        
        $this->stats['websites']['valid']++;
        return 'valid';
    }
    
    /**
     * Email validation (same as before)
     */
    private function validateEmail($email) {
        if (empty($email)) return 'empty';
        
        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $this->stats['emails']['invalid']++;
            return 'invalid';
        }
        
        $domain = substr(strrchr($email, "@"), 1);
        
        $suspiciousDomains = ['example.com', 'test.com', 'gmail.com', 'yahoo.com', 'hotmail.com'];
        if (in_array(strtolower($domain), $suspiciousDomains)) {
            $this->stats['emails']['suspicious']++;
            return 'suspicious';
        }
        
        if (!checkdnsrr($domain, 'MX')) {
            $this->stats['emails']['invalid']++;
            return 'no_mx';
        }
        
        $this->stats['emails']['valid']++;
        return 'valid';
    }
    
    /**
     * Phone validation (same as before)
     */
    private function validatePhone($phone) {
        if (empty($phone)) return 'empty';
        
        $digits = preg_replace('/\D/', '', $phone);
        
        if (strlen($digits) == 10) {
            $this->stats['phones']['valid']++;
            return 'valid';
        } elseif (strlen($digits) == 11 && substr($digits, 0, 1) == '1') {
            $this->stats['phones']['valid']++;
            return 'valid';
        } elseif (strlen($digits) > 6 && strlen($digits) < 15) {
            $this->stats['phones']['formatted']++;
            return 'formatted';
        } else {
            $this->stats['phones']['invalid']++;
            return 'invalid';
        }
    }
    
    /**
     * Save validation results and create flags for admin review
     */
    private function saveValidationResults($results, $store) {
        // Save main validation results
        $this->db->query("
            INSERT INTO contact_validations 
            (store_id, website_status, email_status, phone_status, zip_status, name_status, validated_at)
            VALUES (?, ?, ?, ?, ?, ?, NOW())
            ON DUPLICATE KEY UPDATE
            website_status = VALUES(website_status),
            email_status = VALUES(email_status),
            phone_status = VALUES(phone_status),
            zip_status = VALUES(zip_status),
            name_status = VALUES(name_status),
            validated_at = NOW()
        ", [
            $results['store_id'],
            $results['website_status'],
            $results['email_status'],
            $results['phone_status'],
            $results['zip_status'],
            $results['name_analysis']['status']
        ]);
        
        // Create validation flags for issues requiring review
        $this->createValidationFlags($results, $store);
    }
    
    /**
     * Create flags for admin review
     */
    private function createValidationFlags($results, $store) {
        $flags = [];
        
        // Flag invalid contacts
        if (in_array($results['website_status'], ['invalid', 'no_dns', 'unreachable'])) {
            $flags[] = ['type' => 'website', 'issue' => $results['website_status'], 'current_value' => $store['website']];
        }
        
        if (in_array($results['email_status'], ['invalid', 'no_mx', 'suspicious'])) {
            $flags[] = ['type' => 'email', 'issue' => $results['email_status'], 'current_value' => $store['email']];
        }
        
        if ($results['phone_status'] === 'invalid') {
            $flags[] = ['type' => 'phone', 'issue' => $results['phone_status'], 'current_value' => $store['phone']];
        }
        
        if (in_array($results['zip_status'], ['invalid', 'empty'])) {
            $flags[] = ['type' => 'zip', 'issue' => $results['zip_status'], 'current_value' => $store['postal_code']];
        }
        
        // Flag name issues with suggestions
        if ($results['name_analysis']['status'] === 'flagged') {
            foreach ($results['name_analysis']['issues'] as $issue) {
                $suggestion = '';
                if (isset($results['name_analysis']['suggestions'])) {
                    $suggestion = json_encode($results['name_analysis']['suggestions']);
                }
                
                $flags[] = [
                    'type' => 'name', 
                    'issue' => $issue, 
                    'current_value' => $store['name'], 
                    'suggested_value' => $suggestion
                ];
            }
        }
        
        // Insert flags
        foreach ($flags as $flag) {
            $this->db->query("
                INSERT INTO validation_flags 
                (store_id, flag_type, issue_type, current_value, suggested_value, created_at)
                VALUES (?, ?, ?, ?, ?, NOW())
                ON DUPLICATE KEY UPDATE
                suggested_value = VALUES(suggested_value),
                updated_at = NOW()
            ", [
                $results['store_id'],
                $flag['type'],
                $flag['issue'],
                $flag['current_value'],
                $flag['suggested_value'] ?? null
            ]);
        }
    }
    
    /**
     * Create necessary database tables
     */
    private function createValidationTables() {
        // Contact validations table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS contact_validations (
                id INT AUTO_INCREMENT PRIMARY KEY,
                store_id VARCHAR(100) NOT NULL,
                website_status VARCHAR(20),
                email_status VARCHAR(20),
                phone_status VARCHAR(20),
                zip_status VARCHAR(20),
                name_status VARCHAR(20),
                validated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                UNIQUE KEY unique_store (store_id)
            )
        ");
        
        // Validation flags table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS validation_flags (
                id INT AUTO_INCREMENT PRIMARY KEY,
                store_id VARCHAR(100) NOT NULL,
                flag_type VARCHAR(20) NOT NULL,
                issue_type VARCHAR(30) NOT NULL,
                current_value TEXT,
                suggested_value TEXT,
                status ENUM('pending', 'approved', 'rejected', 'fixed') DEFAULT 'pending',
                admin_notes TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                reviewed_by INT DEFAULT NULL,
                INDEX idx_store_status (store_id, status),
                INDEX idx_type_status (flag_type, status)
            )
        ");
    }
    
    /**
     * Print progress
     */
    private function printProgress() {
        echo sprintf("\n--- Progress: %d stores checked ---\n", $this->stats['total_checked']);
        echo sprintf("Websites: %d valid, %d invalid, %d timeout\n", 
            $this->stats['websites']['valid'],
            $this->stats['websites']['invalid'],
            $this->stats['websites']['timeout']
        );
        echo sprintf("Emails: %d valid, %d invalid, %d suspicious\n",
            $this->stats['emails']['valid'],
            $this->stats['emails']['invalid'],
            $this->stats['emails']['suspicious']
        );
        echo sprintf("Phones: %d valid, %d invalid, %d formatted\n",
            $this->stats['phones']['valid'],
            $this->stats['phones']['invalid'],
            $this->stats['phones']['formatted']
        );
        echo sprintf("Zip Codes: %d valid, %d invalid, %d empty\n",
            $this->stats['zip_codes']['valid'],
            $this->stats['zip_codes']['invalid'],
            $this->stats['zip_codes']['empty']
        );
        echo sprintf("Names: %d valid, %d need cleanup, %d too long\n\n",
            $this->stats['names']['valid'],
            $this->stats['names']['needs_cleanup'],
            $this->stats['names']['too_long']
        );
    }
    
    /**
     * Print final statistics
     */
    private function printFinalStats() {
        echo "\n" . str_repeat('=', 60) . "\n";
        echo "VERIFICATION LIGHT COMPLETE\n";
        echo str_repeat('=', 60) . "\n";
        echo sprintf("Total stores checked: %d\n", $this->stats['total_checked']);
        
        foreach ($this->stats as $category => $data) {
            if ($category === 'total_checked') continue;
            
            echo "\n" . ucfirst(str_replace('_', ' ', $category)) . ":\n";
            foreach ($data as $status => $count) {
                echo sprintf("  %s: %d\n", ucfirst(str_replace('_', ' ', $status)), $count);
            }
        }
        
        echo "\nAdmin review page: " . $this->getAdminUrl() . "\n";
    }
    
    /**
     * Generate summary report for failed validations
     */
    private function generateSummaryReport() {
        $failedCount = $this->db->query("
            SELECT COUNT(*) as count 
            FROM validation_flags 
            WHERE status = 'pending'
        ")->getRowArray();
        
        echo sprintf("\n🚨 %d issues flagged for admin review\n", $failedCount['count']);
    }
    
    /**
     * Get admin interface URL
     */
    private function getAdminUrl() {
        return "https://" . $_SERVER['HTTP_HOST'] . "/tools/verification_admin.php";
    }
    
    /**
     * Generate detailed report
     */
    public function generateDetailedReport() {
        return $this->db->query("
            SELECT 
                cv.*,
                s.name,
                s.website,
                s.email,
                s.phone,
                s.postal_code,
                COUNT(vf.id) as flag_count
            FROM contact_validations cv
            INNER JOIN store s ON cv.store_id = s.store_id
            LEFT JOIN validation_flags vf ON cv.store_id = vf.store_id AND vf.status = 'pending'
            GROUP BY cv.store_id
            ORDER BY flag_count DESC, cv.validated_at DESC
        ")->getResultArray();
    }
}

// Web interface for admin review
if (!isset($_GET['action']) && php_sapi_name() !== 'cli') {
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title>Verification Light Admin</title>
        <style>
            body { font-family: Arial; margin: 20px; }
            .flag { background: #fff3cd; padding: 15px; margin: 10px 0; border-radius: 5px; border-left: 5px solid #ffc107; }
            .flag.name { border-left-color: #007bff; background: #e3f2fd; }
            .flag.website { border-left-color: #dc3545; background: #f8d7da; }
            .flag.email { border-left-color: #fd7e14; background: #fff3cd; }
            .flag.phone { border-left-color: #28a745; background: #d4edda; }
            .suggestion { background: #e8f5e8; padding: 10px; margin: 5px 0; border-radius: 3px; }
            .buttons { margin: 10px 0; }
            button { margin: 5px; padding: 8px 15px; border: none; border-radius: 3px; cursor: pointer; }
            .approve { background: #28a745; color: white; }
            .reject { background: #dc3545; color: white; }
            .edit { background: #007bff; color: white; }
            .stats { background: #f8f9fa; padding: 20px; margin: 20px 0; border-radius: 5px; }
        </style>
    </head>
    <body>
        <h1>Verification Light - Admin Review</h1>
        
        <?php
        $validator = new VerificationLight();
        
        // Handle admin actions
        if (isset($_POST['action'])) {
            $db = \Config\Database::connect();
            
            switch ($_POST['action']) {
                case 'approve_suggestion':
                    // Apply the suggestion and mark as approved
                    $flagId = $_POST['flag_id'];
                    $suggestedValue = $_POST['suggested_value'];
                    
                    // Update the store record
                    // This would need specific field mapping
                    
                    // Mark flag as approved
                    $db->query("UPDATE validation_flags SET status = 'approved' WHERE id = ?", [$flagId]);
                    echo "<div class='alert alert-success'>Suggestion approved and applied!</div>";
                    break;
                    
                case 'reject_flag':
                    $flagId = $_POST['flag_id'];
                    $db->query("UPDATE validation_flags SET status = 'rejected' WHERE id = ?", [$flagId]);
                    echo "<div class='alert alert-info'>Flag rejected.</div>";
                    break;
            }
        }
        
        // Show validation statistics
        $stats = $validator->db->query("
            SELECT 
                flag_type,
                issue_type,
                COUNT(*) as count,
                status
            FROM validation_flags 
            WHERE status = 'pending'
            GROUP BY flag_type, issue_type, status
            ORDER BY count DESC
        ")->getResultArray();
        
        echo "<div class='stats'>";
        echo "<h3>Pending Issues Summary</h3>";
        foreach ($stats as $stat) {
            echo "<p><strong>{$stat['flag_type']} - {$stat['issue_type']}</strong>: {$stat['count']} issues</p>";
        }
        echo "</div>";
        
        // Show flagged businesses
        $flags = $validator->db->query("
            SELECT 
                vf.*,
                s.name,
                s.website,
                s.email,
                s.phone,
                s.postal_code
            FROM validation_flags vf
            INNER JOIN store s ON vf.store_id = s.store_id
            WHERE vf.status = 'pending'
            ORDER BY vf.created_at DESC
            LIMIT 50
        ")->getResultArray();
        
        foreach ($flags as $flag) {
            echo "<div class='flag {$flag['flag_type']}'>";
            echo "<h4>{$flag['name']}</h4>";
            echo "<p><strong>Issue:</strong> {$flag['flag_type']} - {$flag['issue_type']}</p>";
            echo "<p><strong>Current:</strong> {$flag['current_value']}</p>";
            
            if (!empty($flag['suggested_value'])) {
                $suggestions = json_decode($flag['suggested_value'], true);
                if (is_array($suggestions)) {
                    echo "<div class='suggestion'>";
                    echo "<strong>Suggestions:</strong><br>";
                    foreach ($suggestions as $type => $value) {
                        echo "<strong>" . ucwords(str_replace('_', ' ', $type)) . ":</strong> {$value}<br>";
                    }
                    echo "</div>";
                } else {
                    echo "<div class='suggestion'><strong>Suggested:</strong> {$flag['suggested_value']}</div>";
                }
            }
            
            echo "<div class='buttons'>";
            echo "<form method='post' style='display:inline;'>";
            echo "<input type='hidden' name='flag_id' value='{$flag['id']}'>";
            echo "<input type='hidden' name='suggested_value' value='{$flag['suggested_value']}'>";
            echo "<button type='submit' name='action' value='approve_suggestion' class='approve'>Apply Suggestion</button>";
            echo "<button type='submit' name='action' value='reject_flag' class='reject'>Reject</button>";
            echo "<button type='button' class='edit' onclick='editBusiness(\"{$flag['store_id']}\")'>Edit Manually</button>";
            echo "</form>";
            echo "</div>";
            echo "</div>";
        }
        ?>
        
        <script>
        function editBusiness(storeId) {
            window.open('/admin/stores/edit/' + storeId, '_blank');
        }
        </script>
        
    </body>
    </html>
    <?php
    exit;
}

// Command line execution
if (php_sapi_name() === 'cli') {
    $validator = new VerificationLight();
    $batchSize = isset($argv[1]) ? (int)$argv[1] : 50;
    $validator->validateAllContacts($batchSize);
} else {
    echo "<h1>Verification Light</h1>";
    echo "<p>Run from command line: <code>php verification_light.php [batch_size]</code></p>";
    echo "<p><a href='?admin=1'>Admin Review Interface</a></p>";
}
?>