<?php
function analyzeCodeIgniterStructure($dir, $depth = 0) {
    $indent = str_repeat('  ', $depth);
    $files = scandir($dir);
    $output = '';
    
    // Define what we want to focus on
    $coreDirectories = ['app', 'public', 'writable'];
    $importantSubdirs = [
        'Controllers', 'Models', 'Views', 'Config', 'Services', 'Libraries', 
        'Helpers', 'Database', 'Filters'
    ];
    
    // Files we always want to see
    $alwaysShow = [
        '.env', 'composer.json', 'spark', 'index.php', '.htaccess'
    ];
    
    // Extensions we care about for business logic
    $relevantExtensions = ['php', 'sql', 'md'];
    
    // Directories to completely skip
    $skipDirs = [
        'vendor', 'node_modules', 'assets', 'css', 'js', 'fonts', 'images', 
        'scss', 'less', 'libs', 'cache', 'logs', 'session', 'debugbar',
        'builds', 'tests'
    ];
    
    // File patterns to skip
    $skipPatterns = [
        '/\.min\.(js|css)$/',
        '/\.map$/',
        '/\.log$/',
        '/^ci_session/',
        '/\.bak$/',
        '/-backup$/',
        '/-bk$/',
        '/\.zip$/',
        '/\.psd$/',
        '/\.ttf$/',
        '/\.woff$/',
        '/\.eot$/',
        '/\.svg$/',
        '/\.png$/',
        '/\.jpg$/',
        '/\.gif$/',
        '/\.jpeg$/'
    ];
    
    $items = [];
    
    // Categorize and filter files
    foreach ($files as $file) {
        if ($file == '.' || $file == '..') continue;
        
        $path = $dir . '/' . $file;
        $isDir = is_dir($path);
        
        // Skip unwanted directories
        if ($isDir && in_array($file, $skipDirs)) continue;
        
        // Skip files matching unwanted patterns
        $skip = false;
        foreach ($skipPatterns as $pattern) {
            if (preg_match($pattern, $file)) {
                $skip = true;
                break;
            }
        }
        if ($skip && !in_array($file, $alwaysShow)) continue;
        
        // For root level, only show core directories and important files
        if ($depth == 0) {
            if ($isDir && !in_array($file, $coreDirectories)) continue;
            if (!$isDir && !in_array($file, $alwaysShow)) continue;
        }
        
        // For app subdirectories, focus on important ones
        if ($depth == 1 && basename(dirname($path)) == 'app') {
            if ($isDir && !in_array($file, $importantSubdirs)) continue;
        }
        
        // Skip non-relevant file extensions (except for always show files)
        if (!$isDir && !in_array($file, $alwaysShow)) {
            $extension = strtolower(pathinfo($file, PATHINFO_EXTENSION));
            if (!in_array($extension, $relevantExtensions)) continue;
        }
        
        $items[] = [
            'name' => $file,
            'path' => $path,
            'is_dir' => $isDir,
            'mtime' => filemtime($path)
        ];
    }
    
    // Sort: directories first, then files, alphabetically
    usort($items, function($a, $b) {
        if ($a['is_dir'] != $b['is_dir']) {
            return $b['is_dir'] - $a['is_dir']; // Directories first
        }
        return strcasecmp($a['name'], $b['name']);
    });
    
    foreach ($items as $item) {
        $file = $item['name'];
        $path = $item['path'];
        $isDir = $item['is_dir'];
        
        // Add special markers for important files/directories
        $marker = '';
        if ($isDir) {
            // Mark important directories
            $specialDirs = [
                'Controllers' => ' [Business Logic]',
                'Models' => ' [Data Layer]', 
                'Views' => ' [Templates]',
                'Config' => ' [Configuration]',
                'Services' => ' [Business Services]'
            ];
            $marker = isset($specialDirs[$file]) ? $specialDirs[$file] : '';
        } else {
            // Mark important files
            if ($file == 'Routes.php') $marker = ' [URL Routing]';
            if ($file == 'Database.php') $marker = ' [DB Config]';
            if ($file == 'index.php') $marker = ' [Entry Point]';
            if (strpos($file, 'Model.php') !== false) $marker = ' [Data Model]';
            if (strpos($file, 'Controller.php') !== false) $marker = ' [Controller]';
        }
        
        $output .= $indent . ($isDir ? '📁 ' : '📄 ') . $file . $marker . "\n";
        
        // Recurse into directories with depth limits
        if ($isDir) {
            $maxDepth = 4;
            
            // Special depth rules for different directories
            if ($file == 'Views' || $file == 'Controllers') {
                $maxDepth = 3; // Show subdirectory structure for these
            }
            
            if ($depth < $maxDepth) {
                $output .= analyzeCodeIgniterStructure($path, $depth + 1);
            }
        }
    }
    
    return $output;
}

function showProjectSummary($dir) {
    $summary = "🏗️  RINGGOLDROAD BUSINESS DIRECTORY - PROJECT CONTEXT\n";
    $summary .= str_repeat('=', 70) . "\n\n";
    
    $summary .= "📖 WHAT IT IS:\n";
    $summary .= "Sophisticated CodeIgniter 4 business directory with real-time open/closed\n";
    $summary .= "status powered by CRON automation (every 10 minutes), distance-based sorting,\n";
    $summary .= "category management, and professional admin tools. Mobile-optimized with\n";
    $summary .= "toggle filtering and infinite scroll.\n\n";
    
    $summary .= "🏛️ CURRENT ARCHITECTURE STATUS:\n";
    $summary .= "CORE PRINCIPLE: CRON is authoritative for status - never manual time\n";
    $summary .= "calculations. Multiple service types per business (main hours, drive-thru,\n";
    $summary .= "phone support, etc.) with custom admin-defined service names. Category-driven\n";
    $summary .= "organization with flexible many-to-many relationships.\n\n";
    
    $summary .= "✅ PRODUCTION READY: Complete CRUD for stores/categories/hours, sophisticated\n";
    $summary .= "admin interface, real-time status system, verification tools, Excel export\n";
    $summary .= "functionality. Toggle filtering works perfectly with distance-based sorting\n";
    $summary .= "maintained.\n\n";
    
    $summary .= "🔑 KEY TECHNICAL NOTES:\n";
    $summary .= "• Database: store_current_status table is CRON-updated and authoritative\n";
    $summary .= "• Open status: ANY service with status IN ('open', 'closing_soon')\n";
    $summary .= "• Admin interface: Complete category management, store editing, hours config\n";
    $summary .= "• Frontend: Toggle shows only open businesses while maintaining distance sorting\n\n";
    
    $summary .= "🤝 COLLABORATION RULES:\n";
    $summary .= "1. No coding without collaboration - agree on purpose before writing code\n";
    $summary .= "2. Challenge and question - don't agree by default, suggest better alternatives\n";
    $summary .= "3. Deliver whole files when asked - no placeholders or partial code\n";
    $summary .= "4. Check if files exist before creating - avoid redundant development\n";
    $summary .= "5. Suggest improvements - point out features that could be enhanced\n\n";
    
    // Add project log if it exists
    $logFile = dirname($dir) . '/writable/project_log.txt';
    if (file_exists($logFile)) {
        $logContent = file_get_contents($logFile);
        if (!empty(trim($logContent))) {
            $summary .= "📋 PROJECT LOG - RECENT WORK:\n";
            $summary .= str_repeat('-', 40) . "\n";
            $summary .= $logContent . "\n";
        }
    }
    
    // Analyze key components
    $controllerDir = $dir . '/app/Controllers';
    $modelDir = $dir . '/app/Models';
    
    if (is_dir($controllerDir)) {
        $controllers = array_filter(scandir($controllerDir), function($f) use ($controllerDir) {
            return is_dir($controllerDir . '/' . $f) && $f != '.' && $f != '..';
        });
        $summary .= "🎮 Controller Modules: " . implode(', ', $controllers) . "\n";
    }
    
    if (is_dir($modelDir)) {
        $models = array_filter(scandir($modelDir), function($f) {
            return pathinfo($f, PATHINFO_EXTENSION) == 'php' && $f != '.' && $f != '..';
        });
        $summary .= "💾 Data Models: " . count($models) . " models\n";
    }
    
    $summary .= "\n";
    
    return $summary;
}

// Get the project root (parent of document root)
$projectRoot = dirname($_SERVER['DOCUMENT_ROOT']);
$logFile = $projectRoot . '/writable/project_log.txt';

// Handle log entry submission
if (isset($_POST['log_entry']) && !empty(trim($_POST['log_entry']))) {
    $timestamp = date('Y-m-d H:i');
    $entry = '[' . $timestamp . '] ' . trim($_POST['log_entry']) . "\n";
    file_put_contents($logFile, $entry, FILE_APPEND | LOCK_EX);
    header('Location: ' . $_SERVER['PHP_SELF'] . '?logged=1');
    exit;
}

// Check if download is requested
$download = isset($_GET['download']) && $_GET['download'] == 'true';

if ($download) {
    // Generate the analysis content
    $content = showProjectSummary($projectRoot);
    $content .= "📂 PROJECT STRUCTURE:\n";
    $content .= str_repeat('-', 30) . "\n";
    $content .= analyzeCodeIgniterStructure($projectRoot, 0);
    $content .= "\n\n💡 ANALYSIS COMPLETE\n";
    $content .= "This structure focuses on the core CodeIgniter architecture and business logic.\n";
    $content .= "Asset files, vendor libraries, and temporary files have been filtered out.\n";
    
    // Set headers for file download
    $filename = 'ringgoldroad_structure_' . date('Y-m-d_H-i-s') . '.txt';
    header('Content-Type: application/octet-stream');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Content-Length: ' . strlen($content));
    header('Cache-Control: must-revalidate');
    header('Pragma: public');
    
    echo $content;
    exit;
} else {
    // Regular display mode with download option
    header('Content-Type: text/html; charset=utf-8');
    ?>
    <!DOCTYPE html>
    <html>
    <head>
        <title>RinggoldRoad Project Structure</title>
        <style>
            body { font-family: 'Courier New', monospace; margin: 20px; background: #f5f5f5; }
            .container { background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
            .download-section { background: #e3f2fd; padding: 15px; border-radius: 5px; margin-bottom: 20px; }
            .download-btn { 
                background: #2196F3; color: white; padding: 10px 20px; 
                text-decoration: none; border-radius: 5px; font-weight: bold;
                display: inline-block; margin-right: 10px;
            }
            .download-btn:hover { background: #1976D2; }
            pre { background: #f8f9fa; padding: 15px; border-radius: 5px; overflow-x: auto; }
            .info { color: #666; font-size: 14px; }
        </style>
    </head>
    <body>
        <div class="container">
            <h1>🏗️ RinggoldRoad Project Structure Analyzer</h1>
            
            <div class="download-section">
                <h3>📥 Download Options</h3>
                <a href="?download=true" class="download-btn">Download Structure Report</a>
                <span class="info">Get a timestamped .txt file</span>
            </div>
            
            <div class="log-section" style="background: #f3e5f5; padding: 15px; border-radius: 5px; margin-bottom: 20px;">
                <h3>📝 Add Project Log Entry</h3>
                <?php if (isset($_GET['logged'])): ?>
                    <div style="color: green; margin-bottom: 10px;">✅ Log entry added successfully!</div>
                <?php endif; ?>
                <form method="POST" style="margin-bottom: 10px;">
                    <textarea name="log_entry" placeholder="What did we work on today? This will be timestamped and added to future downloads..." 
                              style="width: 100%; height: 80px; padding: 10px; border: 1px solid #ddd; border-radius: 4px; font-family: inherit;"></textarea>
                    <button type="submit" style="background: #9C27B0; color: white; padding: 8px 16px; border: none; border-radius: 4px; margin-top: 5px; cursor: pointer;">
                        Add to Project Log
                    </button>
                </form>
                <div class="info">Entries use format: [2025-07-11 14:30] Your notes here</div>
            </div>
            
            <div class="analysis-output">
                <h3>📊 Live Analysis</h3>
                <pre><?php
                echo htmlspecialchars(showProjectSummary($projectRoot));
                echo "📂 PROJECT STRUCTURE:\n";
                echo str_repeat('-', 30) . "\n";
                echo htmlspecialchars(analyzeCodeIgniterStructure($projectRoot, 0));
                echo "\n\n💡 ANALYSIS COMPLETE\n";
                echo "This structure focuses on the core CodeIgniter architecture and business logic.\n";
                echo "Asset files, vendor libraries, and temporary files have been filtered out.\n";
                ?></pre>
            </div>
            
            <div class="info" style="margin-top: 20px;">
                <strong>Usage:</strong> Click "Download Structure Report" to get a .txt file with timestamp that you can easily share with Claude for project analysis.
            </div>
        </div>
    </body>
    </html>
    <?php
}
?>